
C====================================================================
C     SUBROUTINE GAMMA
C     Purpose: Computes the Euler gamma function of X
C====================================================================
      SUBROUTINE GAMMA(X,GA)

        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION G(26)
	COMMON/PARAMETERS/ PI
	
        IF (X.EQ.INT(X)) THEN
           IF (X.GT.0.0D0) THEN
              GA=1.0D0
              M1=X-1
              DO 10 K=2,M1
10               GA=GA*K
           ELSE
              GA=1.0D+300
           ENDIF
        ELSE
           IF (DABS(X).GT.1.0D0) THEN
              Z=DABS(X)
              M=INT(Z)
              R=1.0D0
              DO 15 K=1,M
15               R=R*(Z-K)
              Z=Z-M
           ELSE
              Z=X
           ENDIF
           DATA G/1.0D0,0.5772156649015329D0,
     &          -0.6558780715202538D0,-0.420026350340952D-1,
     &          0.1665386113822915D0,-.421977345555443D-1,
     &          -.96219715278770D-2, .72189432466630D-2,
     &          -.11651675918591D-2, -.2152416741149D-3,
     &          .1280502823882D-3, -.201348547807D-4,
     &          -.12504934821D-5, .11330272320D-5,
     &          -.2056338417D-6, .61160950D-8,
     &          .50020075D-8, -.11812746D-8,
     &          .1043427D-9, .77823D-11,
     &          -.36968D-11, .51D-12,
     &          -.206D-13, -.54D-14, .14D-14, .1D-15/
           GR=G(26)
           DO 20 K=25,1,-1
20            GR=GR*Z+G(K)
           GA=1.0D0/(GR*Z)
           IF (DABS(X).GT.1.0D0) THEN
              GA=GA*R
              IF (X.LT.0.0D0) GA=-PI/(X*GA*DSIN(PI*X))
           ENDIF
        ENDIF
        RETURN
        END SUBROUTINE GAMMA
C====================================================================
C     FUNCTION GM 
C     Purpose: Computes the BH cross section gamma factor
C====================================================================
        DOUBLEPRECISION FUNCTION GGM()

        IMPLICIT NONE
        INTEGER NI,MOMTRANSF
	DOUBLE PRECISION GA,N,MST,ALPHA,XMIN,QMIN
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

        N=DBLE(NI)
	CALL GAMMA((N+3.0D0)/2.0D0,GA)
        GGM=(8.0D0*GA/(N+2.0D0))**(1.0D0/(N+1.0D0))

        RETURN
        END FUNCTION GGM
C====================================================================
C     FUNCTION CROSS 
C     Purpose: Computes the BH cross section
C====================================================================
      DOUBLEPRECISION FUNCTION CROSS(S)

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF
      DOUBLE PRECISION MST,S,ALPHA,XMIN,QMIN,REALBMAX,SST,PI,N,LAMBDA
      EXTERNAL REALBMAX,LAMBDA
      LOGICAL YNFLAG,EMCHARGEFLAG
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
      COMMON/PARAMETERS/ PI
      
      IF (YNFLAG .EQV. .FALSE.) THEN
        SST=MST**2.0D0
	N=DBLE(NI)
        CROSS=PI*LAMBDA()**2.0D0*(S/SST)**(1.0D0/(N+1.0D0))/SST
      ELSE
        CROSS=PI*REALBMAX(S)*REALBMAX(S)	
      ENDIF
      
      RETURN
      END FUNCTION CROSS     
C====================================================================
C     FUNCTION  FF 
C     Purpose: Returns the Yoshino-Nambu form factor
C====================================================================
      DOUBLEPRECISION FUNCTION FF()

      IMPLICIT NONE
      INTEGER NI,UP,YVER,MOMTRANSF
      DOUBLE PRECISION MST,ALPHA,XMIN,QMIN,ZYN(100),YYN(100)
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON /YOSHINO/ ZYN,YYN,UP,YVER

      IF (YVER .EQ. 0) THEN     
      
        IF(NI .EQ. 3) THEN
           FF=1.515D0
        ELSE IF(NI .EQ. 4) THEN
           FF=1.642D0
        ELSE IF(NI .EQ. 5) THEN
           FF=1.741D0
        ELSE IF(NI .EQ. 6) THEN
           FF=1.819D0
        ELSE IF(NI .EQ. 7)  THEN
           FF=1.883D0
        ENDIF
	
      ELSE
      		
        IF(NI .EQ. 3) THEN
           FF=2.52D0
        ELSE IF(NI .EQ. 4) THEN
           FF=2.77D0
        ELSE IF(NI .EQ. 5) THEN
           FF=2.95D0
        ELSE IF(NI .EQ. 6) THEN
           FF=3.09D0
        ELSE IF(NI .EQ. 7)  THEN
           FF=3.20D0
        ENDIF
	
      ENDIF	

      RETURN
      END FUNCTION FF
C====================================================================
C     FUNCTION TEVTOPICO
C     Purpose: Changes units from Tev^{-2} to picobarn
C====================================================================
      DOUBLEPRECISION FUNCTION TEVTOPICO(CRF)

      IMPLICIT NONE
      DOUBLE PRECISION CRF
      
      TEVTOPICO=19.733D0**2.0D0*CRF

      RETURN
      END FUNCTION TEVTOPICO
C====================================================================
C     FUNCTION OMEGA 
C     Purpose: Computes the volume of the sphere
C====================================================================  
      DOUBLEPRECISION FUNCTION OMEGA(DI)

      IMPLICIT NONE
      INTEGER DI
      DOUBLE PRECISION GA,PI,D
      COMMON/PARAMETERS/ PI

      D=DBLE(DI)
      CALL GAMMA((D+1.0D0)/2.0D0,GA)
      OMEGA=2.0D0*PI**((D+1.0D0)/2.0D0)/GA
      RETURN
      END FUNCTION OMEGA   
C====================================================================  
C     FUNCTION RANDOM
C     Purpose: Generates a random number
C====================================================================
      DOUBLEPRECISION FUNCTION RANDOM() 

      IMPLICIT NONE
      CHARACTER C 
      DOUBLEPRECISION A 
      PARAMETER(A=1.0D0/256.0D0) 
      INTEGER I

      RANDOM = 0.0D0 
      DO I = 1, 8 
        CALL FGETC(1, C)
        RANDOM = (RANDOM + ICHAR(C)) * A
      ENDDO
      RETURN
      END FUNCTION RANDOM
C====================================================================
C     FUNCTION SCHWARZSCHILD 
C     Purpose: Computes the Schwarzschild radius of the BH 
C====================================================================
	DOUBLEPRECISION FUNCTION SCHWARZSCHILD(B,S)

	IMPLICIT NONE
        INTEGER NI,MOMTRANSF
	DOUBLE PRECISION D,MST,LAMBDA,M,ALPHA,B,MBH,XMIN,QMIN,S
	EXTERNAL LAMBDA,MBH
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

        D=DBLE(NI+4)
	M=MBH(B,S)/MST
	SCHWARZSCHILD=LAMBDA()*M**(1.0D0/(D-3.0D0))/MST

	RETURN
	END FUNCTION SCHWARZSCHILD
C====================================================================
C     FUNCTION LAMBDA 
C     Purpose: Computes the dimensionless area factor lambda
C====================================================================
	DOUBLEPRECISION FUNCTION LAMBDA()

	IMPLICIT NONE
	INTEGER NI,MOMTRANSF
	DOUBLE PRECISION N,PI,MST,ALPHA,OMEGA,XMIN,QMIN
	EXTERNAL OMEGA
	COMMON/PARAMETERS/ PI
        COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

	N=DBLE(NI)
	LAMBDA=(16.0D0*PI/((N+2.0D0)*OMEGA(NI+2)))**(1.0D0/(N+1.0D0))	

	RETURN
	END FUNCTION LAMBDA
C====================================================================
C     FUNCTION TEMPERATURE 
C     Purpose: Computes the BH temperature
C====================================================================
	DOUBLEPRECISION FUNCTION TEMPERATURE(B,S)

        IMPLICIT NONE
	INTEGER NI,MOMTRANSF
	DOUBLE PRECISION ALPHA,MST,M,LAMBDA,PI,D,MBH,B,XMIN,QMIN,S
	EXTERNAL LAMBDA,MBH
	COMMON/PARAMETERS/ PI
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

	D=DBLE(NI+4)	
	M=MBH(B,S)/MST
	TEMPERATURE=((D-3.0D0)/(4.0D0*PI*LAMBDA()))*
     !	MST*M**(1.0D0/(3.0D0-D))
        IF (ALPHA .EQ. 0.0D0) THEN
	  GOTO 100
	ELSE 
          TEMPERATURE=2.0D0*TEMPERATURE/(1.0D0+DSQRT(1.0D0-
     !	  (ALPHA/LAMBDA())**2.0D0*M**(2.0D0/(3.0D0-D))))
        ENDIF
	
100	RETURN
	END FUNCTION TEMPERATURE
C====================================================================
C     FUNCTION MINMASS
C     Purpose: Computes the minimum BH mass with GUP effects
C====================================================================
      DOUBLEPRECISION FUNCTION MINMASS()

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF
      DOUBLE PRECISION D,ALPHA,MST,GA,PI,XMIN,QMIN
      COMMON/PARAMETERS/ PI
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      D=DBLE(NI+4)
      CALL GAMMA((D-1.0D0)/2.0D0,GA)
      MINMASS=(D-2.0D0)*((ALPHA*DSQRT(PI))**(D-3.0D0))*
     ! MST/8.0D0/GA
     
      RETURN
      END FUNCTION MINMASS
C====================================================================
C     FUNCTION DOFS  
C     Purpose: Gives the degrees of freedom
C====================================================================
      DOUBLEPRECISION FUNCTION DOFS(SPECIES)

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF,SPECIES
      DOUBLE PRECISION D,MST,ALPHA,XMIN,QMIN
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      D=DBLE(NI+4)

      IF (SPECIES .EQ. 1) THEN
         DOFS=1.0D0
      ELSE IF(SPECIES .EQ. 2) THEN
         DOFS=90.0D0
      ELSE IF(SPECIES .EQ. 3) THEN
         DOFS=27.0D0
      ELSE IF(SPECIES .EQ. 4) THEN
         DOFS=D*(D-3.0D0)/2.0D0
      ELSE
        OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')
        WRITE(99,*) 'Error: Unknown particle species in function DOFS '
        CLOSE(99)
        STOP
      ENDIF	

	RETURN
	END FUNCTION DOFS
C====================================================================
C     SUBROUTINE PMATRIX  
C     Purpose: Computes the particle property matrix
C====================================================================
      SUBROUTINE PMATRIX(PARTICLE)

      IMPLICIT NONE
      DOUBLE PRECISION PARTICLE(4,7,2),DOFS,NRM(4,2)
      EXTERNAL DOFS

	NRM(1,1)=1.4D-3
	NRM(2,1)=4.8D-4
	NRM(3,1)=1.5D-4
	NRM(4,1)=2.2D-5
	NRM(1,2)=2.9D-4
	NRM(2,2)=1.6D-4
	NRM(3,2)=6.7D-5
	NRM(4,2)=1.5D-5
		
	PARTICLE(1,1,1)=4.75D0*NRM(1,1)
	PARTICLE(1,2,1)=13.0D0*NRM(1,1)
	PARTICLE(1,3,1)=27.4D0*NRM(1,1)
	PARTICLE(1,4,1)=49.3D0*NRM(1,1)
	PARTICLE(1,5,1)=79.9D0*NRM(1,1)
	PARTICLE(1,6,1)=121.0D0*NRM(1,1)
	PARTICLE(1,7,1)=172.0D0*NRM(1,1)

	PARTICLE(1,1,2)=8.94D0*NRM(1,2)
	PARTICLE(1,2,2)=36.0D0*NRM(1,2)
	PARTICLE(1,3,2)=99.8D0*NRM(1,2)
	PARTICLE(1,4,2)=222.0D0*NRM(1,2)
	PARTICLE(1,5,2)=429.0D0*NRM(1,2)
	PARTICLE(1,6,2)=749.0D0*NRM(1,2)
	PARTICLE(1,7,2)=1220.0D0*NRM(1,2)

	PARTICLE(2,1,1)=9.05D0*NRM(2,1)
	PARTICLE(2,2,1)=27.6D0*NRM(2,1)
	PARTICLE(2,3,1)=58.2D0*NRM(2,1)
	PARTICLE(2,4,1)=103.0D0*NRM(2,1)
	PARTICLE(2,5,1)=163.0D0*NRM(2,1)
	PARTICLE(2,6,1)=240.0D0*NRM(2,1)
	PARTICLE(2,7,1)=335.0D0*NRM(2,1)

	PARTICLE(2,1,2)=14.2D0*NRM(2,2)
	PARTICLE(2,2,2)=59.5D0*NRM(2,2)
	PARTICLE(2,3,2)=162.0D0*NRM(2,2)
	PARTICLE(2,4,2)=352.0D0*NRM(2,2)
	PARTICLE(2,5,2)=664.0D0*NRM(2,2)
	PARTICLE(2,6,2)=1140.0D0*NRM(2,2)
	PARTICLE(2,7,2)=1830.0D0*NRM(2,2)

	PARTICLE(3,1,1)=19.2D0*NRM(3,1)
	PARTICLE(3,2,1)=80.6D0*NRM(3,1)
	PARTICLE(3,3,1)=204.0D0*NRM(3,1)
	PARTICLE(3,4,1)=403.0D0*NRM(3,1)
	PARTICLE(3,5,1)=689.0D0*NRM(3,1)
	PARTICLE(3,6,1)=1070.0D0*NRM(3,1)
	PARTICLE(3,7,1)=1560.0D0*NRM(3,1)

	PARTICLE(3,1,2)=27.1D0*NRM(3,2)
	PARTICLE(3,2,2)=144.0D0*NRM(3,2)
	PARTICLE(3,3,2)=441.0D0*NRM(3,2)
	PARTICLE(3,4,2)=1020.0D0*NRM(3,2)
	PARTICLE(3,5,2)=2000.0D0*NRM(3,2)
	PARTICLE(3,6,2)=3530.0D0*NRM(3,2)
	PARTICLE(3,7,2)=5740.0D0*NRM(3,2)

	PARTICLE(4,1,1)=49.0*NRM(4,1)/DOFS(4)
	PARTICLE(4,2,1)=395.0D0*NRM(4,1)/DOFS(4)
	PARTICLE(4,3,1)=1247.0D0*NRM(4,1)/DOFS(4)
	PARTICLE(4,4,1)=4701.0D0*NRM(4,1)/DOFS(4)
	PARTICLE(4,5,1)=9990.0D0*NRM(4,1)/DOFS(4)
	PARTICLE(4,6,1)=30888.0D0*NRM(4,1)/DOFS(4)
	PARTICLE(4,7,1)=65390.0D0*NRM(4,1)/DOFS(4)

	PARTICLE(4,1,2)=103.0D0*NRM(4,2)/DOFS(4)
	PARTICLE(4,2,2)=1036.0D0*NRM(4,2)/DOFS(4)
	PARTICLE(4,3,2)=5121.0D0*NRM(4,2)/DOFS(4)
	PARTICLE(4,4,2)=2.0D4*NRM(4,2)/DOFS(4)
	PARTICLE(4,5,2)=7.1D4*NRM(4,2)/DOFS(4)
	PARTICLE(4,6,2)=2.5D5*NRM(4,2)/DOFS(4)
	PARTICLE(4,7,2)=8.0D5*NRM(4,2)/DOFS(4)

	RETURN
	END SUBROUTINE PMATRIX
C====================================================================
C     SUBROUTINE SUMMULTI
C     Purpose: Computes the sums of Dof and Greybody
C====================================================================
	SUBROUTINE SUMMULTI(SUMS)

	IMPLICIT NONE
	INTEGER I,NI,MOMTRANSF
	DOUBLE PRECISION MST,ALPHA,SUMS(2),PARTICLE(4,7,2),XMIN,QMIN
	DOUBLE PRECISION DOFS
	EXTERNAL DOFS
        COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
	
	SUMS(1)=0.0D0
	SUMS(2)=0.0D0
	
	CALL PMATRIX(PARTICLE)
	DO I=1,4
	SUMS(1)=SUMS(1)+PARTICLE(I,NI,1)*DOFS(I)
	SUMS(2)=SUMS(2)+PARTICLE(I,NI,2)*DOFS(I)
        ENDDO
        
	RETURN
	END SUBROUTINE SUMMULTI
C====================================================================	
C     FUNCTION MU 
C     Purpose: Computes the Mu factor in the BH lifetime 
C====================================================================
      DOUBLEPRECISION FUNCTION MU()

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF
      DOUBLE PRECISION MST,LAMBDA,D,SUMS(2),XMIN,QMIN,PI,ALPHA
      EXTERNAL LAMBDA
      COMMON/PARAMETERS/ PI
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      D=DBLE(NI+4)

      CALL SUMMULTI(SUMS)
      
      MU=SUMS(2)/LAMBDA()**2.0D0

      RETURN
      END FUNCTION MU
C====================================================================
C     FUNCTION LIFETIME
C     Purpose: Computes the BH lifetime 
C====================================================================
	DOUBLEPRECISION FUNCTION LIFETIME(B,S)	

        IMPLICIT NONE
	INTEGER I,N1,N2,NI,MOMTRANSF
	DOUBLEPRECISION ALPHA,MST,M,LAMBDA,IGUP,X,MU,D,B,MBH,XMIN,QMIN,S
	DOUBLEPRECISION ARGGUP,FPGUP(2),INTEGRATE,DXINIT
	EXTERNAL LAMBDA,MU,MBH,ARGGUP,INTEGRATE
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
        COMMON/GUPINTEGRAL/FPGUP
 
        D=DBLE(NI+4)
	M=MBH(B,S)/MST

	IF (ALPHA .EQ. 0.0D0) THEN
	  LIFETIME=(D-3.0D0)/(D-1.0D0)*M**((D-1.0D0)/
     !	           (D-3.0D0))/MU()/MST
	ELSE 
	    X=LAMBDA()*M**(1.0D0/(D-3.0D0))/ALPHA
	    FPGUP(1)=4.0D0
	    FPGUP(2)=DBLE(NI-2)
	    DXINIT=(x-1.0D0)/1.0D2
	    IGUP=INTEGRATE(ARGGUP,FPGUP,2,1.0D0,X,DXINIT,1.D-5)
            LIFETIME=(D-3.0D0)*(ALPHA/LAMBDA())**(D-1.0D0)
     !	             /16/MU()*IGUP/MST
        ENDIF
        RETURN
	END FUNCTION LIFETIME
C====================================================================
C     FUNCTION ARGGUP     
C====================================================================
      DOUBLEPRECISION FUNCTION ARGGUP(X)
      IMPLICIT NONE 
      DOUBLEPRECISION X,FPGUP(2)
      COMMON/GUPINTEGRAL/FPGUP
            
      ARGGUP=X**(FPGUP(2))*(X+DSQRT(X*X-1.0D0))**(FPGUP(1))

      END FUNCTION ARGGUP
C====================================================================
C     FUNCTION ENTROPY
C     Purpose: Computes the BH entropy 
C====================================================================
      DOUBLEPRECISION FUNCTION ENTROPY(B,S)

      IMPLICIT NONE
      INTEGER NI,I,MOMTRANSF
      DOUBLEPRECISION D,ALPHA,MST,M,LAMBDA,IGUP,PI,X,B,MBH,XMIN,QMIN,S
      DOUBLEPRECISION INTEGRATE,dxinit,fpgup(2),arggup
      EXTERNAL LAMBDA,MBH,INTEGRATE,arggup
      COMMON/PARAMETERS/ PI
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF      
      COMMON/GUPINTEGRAL/FPGUP 
        
      D=DBLE(NI+4)
      M=MBH(B,S)/MST

      IF (ALPHA .EQ. 0.0D0) THEN
	 ENTROPY=(4.0D0*PI*LAMBDA()/(D-2.0D0))*M**((D-2.0D0)/(D-3.0D0))
      ELSE 
	 X=LAMBDA()*M**(1.0D0/(D-3.0D0))/ALPHA	 
	 FPGUP(1)=1.0D0
	 FPGUP(2)=DBLE(NI)
	 DXINIT=(X-1.0D0)/1.0D2
	 IGUP=INTEGRATE(ARGGUP,FPGUP,2,1.0D0,X,DXINIT,1.0D-5)	 
         ENTROPY=2.0D0*PI*LAMBDA()*(ALPHA/LAMBDA())**(D-2.0D0)*IGUP
      ENDIF

      RETURN
      END FUNCTION ENTROPY
C====================================================================	
C     FUNCTION SPECIFICHEAT 
C     Purpose: Computes the BH SPECIFIC HEAT
C====================================================================
	DOUBLEPRECISION FUNCTION SPECIFICHEAT(B,S)

        IMPLICIT NONE
	INTEGER NI,MOMTRANSF
	DOUBLE PRECISION ALPHA,MST,M,LAMBDA,PI,D,B,MBH,XMIN,QMIN,S
	EXTERNAL LAMBDA,MBH
	COMMON/PARAMETERS/ PI
        COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

	D=DBLE(NI+4)	
	M=MBH(B,S)/MST
	SPECIFICHEAT=-2*PI*LAMBDA()*M**((D-2.0D0)/(D-3.0D0))
     !	  *DSQRT(1-(ALPHA/LAMBDA())**2.0D0*M**(2.0D0/(3.0D0-D)))*
     !    (1+DSQRT(1-(ALPHA/LAMBDA())**2.0D0*M**(2.0D0/(3.0D0-D))))
        
	RETURN
	END FUNCTION SPECIFICHEAT
C====================================================================
C     FUNCTION MULTIPLICITY 
C     Purpose: Computes the BH total multiplicity
C====================================================================
	DOUBLEPRECISION FUNCTION MULTIPLICITY(B,S)

        IMPLICIT NONE
	INTEGER NI,MOMTRANSF
	DOUBLE PRECISION ALPHA,MST,PI,B,SUMS(2),XMIN,QMIN,S
	DOUBLE PRECISION ENTROPY,D
 	EXTERNAL ENTROPY
	COMMON/PARAMETERS/ PI
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

	D=DBLE(NI+4)
	
	CALL SUMMULTI(SUMS)

        MULTIPLICITY=(0.25D0*(D-3.0D0)/PI)*ENTROPY(B,S)*SUMS(1)/SUMS(2)
	
	RETURN
	END FUNCTION MULTIPLICITY
C====================================================================
C     SUBROUTINE FLAVOR 
C     Purpose: Computes the spin-flavor multiplicity 
C====================================================================
	SUBROUTINE FLAVOR(S,B,NF)

        IMPLICIT NONE
	INTEGER I,NI,SPECIES,MOMTRANSF
	DOUBLE PRECISION MST,ALPHA,PI,B,XMIN,QMIN,S,DOFS
	DOUBLEPRECISION MULTIPLICITY, NF(4),SUMS(2),PARTICLE(4,7,2)
	EXTERNAL MULTIPLICITY,DOFS
	COMMON/PARAMETERS/ PI
        COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
        
	CALL PMATRIX(PARTICLE)
	CALL SUMMULTI(SUMS)

	DO I=1,4
	NF(I)=MULTIPLICITY(B,S)*PARTICLE(I,NI,1)*DOFS(I)/SUMS(1)
	ENDDO

	RETURN
	END SUBROUTINE FLAVOR
C====================================================================	
C     FUNCTION YN  
C     Purpose: Reads the YN tables
C==================================================================== 
      SUBROUTINE YN()

      IMPLICIT NONE
      INTEGER NI,J,UP,YVER,MOMTRANSF
      DOUBLEPRECISION MST,ALPHA,ZYN(100),YYN(100),XMIN,QMIN
      CHARACTER NAMEFILE*12,NCHAR*1
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON /YOSHINO/ ZYN,YYN,UP,YVER

      IF (YVER .EQ. 0) THEN
      
        IF (NI .EQ. 7) THEN
           UP=75
        ELSE IF (NI .EQ. 6) THEN
           UP=71
        ELSE IF (NI .EQ. 5) THEN
           UP=71
        ELSE IF (NI .EQ. 4) THEN
           UP=65
        ELSE IF (NI .EQ. 3) THEN
           UP=70
        ENDIF 

        NCHAR=CHAR(NI+48)
        NAMEFILE='Yoshino'//NCHAR//'.dat'
        OPEN(UNIT=1,FILE=NAMEFILE,STATUS='OLD',ERR=3000)		
        DO 5 J=1, UP
        READ(1,1100) ZYN(J), YYN(J)
5       CONTINUE	
        CLOSE(1)
      
      ELSE 

        IF (NI .EQ. 7) THEN
           UP=49
        ELSE IF (NI .EQ. 6) THEN
           UP=41
        ELSE IF (NI .EQ. 5) THEN
           UP=47
        ELSE IF (NI .EQ. 4) THEN
           UP=41
        ELSE IF (NI .EQ. 3) THEN
           UP=42
        ENDIF 

        NCHAR=CHAR(NI+48)
        NAMEFILE='YR'//NCHAR//'.dat'
        OPEN(UNIT=1,FILE=NAMEFILE,STATUS='OLD',ERR=3000)		
        DO 10 J=1, UP
        READ(1,1200) ZYN(J), YYN(J)
10      CONTINUE	
        CLOSE(1)

      ENDIF	      

1000    format (8(1pe13.4))
1100	format (4(f15.7))		
1200	format (4(f18.7))		

      RETURN
3000  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Error: Damaged or missing graviton loss data file]'
      CLOSE(99)
      STOP       
      END SUBROUTINE YN
C==================================================================== 
C     FUNCTION BMAX
C     Purpose: Computes the maximum impact parameter of Yoshino-Nambu
C==================================================================== 
       DOUBLEPRECISION FUNCTION BMAX(S)

       IMPLICIT NONE
       INTEGER NI,MOMTRANSF
       DOUBLEPRECISION N,PI,MST,ALPHA,FF,XMIN,QMIN,LAMBDA,S
       EXTERNAL FF,LAMBDA
       COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
       COMMON/PARAMETERS/ PI

       N=DBLE(NI)
       BMAX=DSQRT(FF())*LAMBDA()*(DSQRT(S)/MST)**(1.0D0/(N+1.0D0))/MST
       
       RETURN
       END FUNCTION BMAX
C====================================================================
C     FUNCTION R0
C     Purpose: Computes the r0 of Yoshino-Nambu
C==================================================================== 
       DOUBLEPRECISION FUNCTION R0(S)

       IMPLICIT NONE
       INTEGER NI,MOMTRANSF
       DOUBLEPRECISION N,PI,MST,ALPHA,XMIN,QMIN,S,OMEGA
       EXTERNAL OMEGA
       COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
       COMMON/PARAMETERS/ PI

       N=DBLE(NI)
       R0=(4.0d0*PI*DSQRT(S)/MST/OMEGA(NI+1))**(1.0D0/(N+1.0D0))/MST

       RETURN
       END FUNCTION R0
C====================================================================
C     FUNCTION FTHRESHOLD
C     Purpose: Computes the BH threshold formation
C==================================================================== 
      DOUBLEPRECISION FUNCTION FTHRESHOLD()

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF
      DOUBLEPRECISION MST,ALPHA,MINMASS,XMIN,QMIN
      EXTERNAL MINMASS
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      IF (MINMASS() .GT. MST ) THEN 
      FTHRESHOLD=XMIN*MINMASS()
      ELSE 
      FTHRESHOLD=XMIN*MST
      ENDIF
      END FUNCTION FTHRESHOLD
C====================================================================
C     FUNCTION QTHRESHOLD
C     Purpose: Computes the BH threshold evaporation
C==================================================================== 
      DOUBLEPRECISION FUNCTION QTHRESHOLD()

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF
      DOUBLEPRECISION MST,ALPHA,MINMASS,XMIN,QMIN
      EXTERNAL MINMASS
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      IF (MINMASS() .GT. MST ) THEN 
      QTHRESHOLD=QMIN*MINMASS()
      ELSE 
      QTHRESHOLD=QMIN*MST
      ENDIF
      END FUNCTION QTHRESHOLD
C====================================================================
C     FUNCTION REALBMAX
C     Purpose: Computes the maximum impact parameter for BH formation
C==================================================================== 
       DOUBLEPRECISION FUNCTION REALBMAX(S)

       IMPLICIT NONE
       INTEGER NI,I,UP,YVER,MOMTRANSF
       DOUBLEPRECISION N,PI,MST,ALPHA,S,OMEGA,XMIN,QMIN,FTHRESHOLD
       DOUBLEPRECISION AA,BB,ZYN(100),YYN(100),DY,BMAX,R0
       EXTERNAL OMEGA,BMAX,FTHRESHOLD,R0
       COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF       
       COMMON /YOSHINO/ ZYN,YYN,UP,YVER

       AA=FTHRESHOLD()/DSQRT(S)
	
	IF (YYN(UP) .GT. AA) THEN
          CALL POLINTT(YYN(UP-2),ZYN(UP-2),3,AA,BB,DY)
	  GOTO 20
        ENDIF
	
        DO I=2,UP-1
	  IF (YYN(I) .LT. AA) THEN
	    CALL POLINTT(YYN(I-1),ZYN(I-1),3,AA,BB,DY)
	    GOTO 20 
	  ENDIF
	ENDDO

20     IF (YVER .EQ. 0) THEN            
         REALBMAX=BB*BMAX(S)       
       ELSE       	  
         REALBMAX=BB*R0(S)
       ENDIF	  	     
	 
       RETURN
       END FUNCTION REALBMAX
C====================================================================
C     FUNCTION MBH
C     Purpose: Computes the BH mass of Yoshino-Nambu
C==================================================================== 
        DOUBLEPRECISION FUNCTION MBH(B,S)

        IMPLICIT NONE
        INTEGER NI,I,UP,YVER,MOMTRANSF
        DOUBLEPRECISION N,PI,MST,ALPHA,S,ZYN(100),YYN(100),BMAX
	DOUBLEPRECISION BNORM,DY,B,MBHNORM,XMIN,QMIN,R0
        LOGICAL YNFLAG,EMCHARGEFLAG
        EXTERNAL BMAX,R0
        COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
	COMMON /YOSHINO/ ZYN,YYN,UP,YVER		
        COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
	
	IF (YNFLAG .EQV. .FALSE.) THEN 
	  MBH=DSQRT(S)
	  GOTO 30
	ELSE

	IF (YVER .EQ. 0) THEN	
	  BNORM=B/BMAX(S)
	ELSE
	  BNORM=B/R0(S)
	ENDIF	
	
	IF (ZYN(UP) .LT. BNORM) THEN
          CALL POLINTT(ZYN(UP-2),YYN(UP-2),3,BNORM,MBHNORM,DY)	     
	  GOTO 20
        ENDIF
	
        DO I=2,UP-1
	  IF (ZYN(I) .GT. BNORM) THEN
	    CALL POLINTT(ZYN(I-1),YYN(I-1),3,BNORM,MBHNORM,DY)
	    GOTO 20 
	  ENDIF
	ENDDO
	
20	CONTINUE
        MBH=MBHNORM*DSQRT(S)
	ENDIF

30      CONTINUE

      RETURN         
      END FUNCTION MBH
C====================================================================
C     SUBROUTINE FRAGMENT
C     Purpose: Fragments the spin-flavor multiplicity
C====================================================================
      SUBROUTINE FFRAGMENT(S,B,NUMPARTICLE,TABLEPARTICLE,TABLEMASS,
     ! HARDPARTICLE,HARDMASS,EMCHARGE)

      IMPLICIT NONE
      INTEGER NI,I,J,NFINT(4),PROB,NUMPARTICLE,MOMTRANSF
      INTEGER TABLEPARTICLE(100,3),EMCHARGE,NP
      INTEGER HARDPARTICLE(18,3) 
      DOUBLEPRECISION PI,MST,ALPHA,B,NF(4),S
      DOUBLEPRECISION MULTIPLICITY,MBH,MINMASS,XMIN,QMIN
      DOUBLEPRECISION QTHRESHOLD,THARDMASS,DOFS
      DOUBLEPRECISION TABLEMASS(100),HARDMASS(18)
      LOGICAL YNFLAG,EMCHARGEFLAG
      EXTERNAL MULTIPLICITY,MBH,QTHRESHOLD,DOFS

      INTEGER KGENEV
      REAL TECM,AMASS(18),PCM(5,18),WT
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      COMMON/GENIN / NP, TECM, AMASS, KGENEV
      COMMON/GENOUT/ PCM, WT
      
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG	

      REAL RVEC

      CALL PARTICLEMATRIX(TABLEPARTICLE,TABLEMASS)
      CALL FLAVOR(S,B,NF)
            
      NUMPARTICLE=0
                      
      DO I=1,4
       NFINT(I)=INT(NF(I)+0.5)
      ENDDO
      
10    DO I=1,100
	  TABLEPARTICLE(I,1)=0
      ENDDO

      DO I=1,NP
        DO J=1,3
	 HARDPARTICLE(I,J)=0
       ENDDO
       HARDMASS(I)=0.0D0       
      ENDDO
      
      THARDMASS=0.0D0
      
      TABLEPARTICLE(30,1)=NFINT(4)
      TABLEPARTICLE(31,1)=NFINT(1)
       
      DO 40 I=1,NFINT(2)

         CALL RANLUX(RVEC,4)
         PROB=INT(RVEC*DOFS(2))

	 DO J=1,12
	  IF (PROB .LT. J*6) THEN
	  TABLEPARTICLE(J,1)=TABLEPARTICLE(J,1)+1
	  GOTO 40
	  ENDIF
	 ENDDO
	 DO J=13,18
	  IF (PROB .LT. 72+(J-12)*2) THEN
	  TABLEPARTICLE(J,1)=TABLEPARTICLE(J,1)+1
	  GOTO 40
	  ENDIF
	 ENDDO
	 DO J=19,24
	  IF (PROB .LT. 84+(J-18)*1) THEN
	  TABLEPARTICLE(J,1)=TABLEPARTICLE(J,1)+1
	  GOTO 40
	  ENDIF
	 ENDDO	
40      CONTINUE

	DO 80 I=1,NFINT(3)

         CALL RANLUX(RVEC,4)        
         PROB=INT(RVEC*DOFS(3))

	  IF (PROB .LT. 16) THEN
	   TABLEPARTICLE(25,1)=TABLEPARTICLE(25,1)+1
	   GOTO 80
	  ELSEIF (PROB .LT. 18) THEN
	   TABLEPARTICLE(26,1)=TABLEPARTICLE(26,1)+1
	   GOTO 80
	  ENDIF
	  DO J=27,29
	   IF (PROB .LT. 18+(J-26)*3) THEN
	    TABLEPARTICLE(J,1)=TABLEPARTICLE(J,1)+1
	    GOTO 80
	   ENDIF
	  ENDDO
80      CONTINUE
                           
      DO I=1,NP

         CALL RANLUX(RVEC,4)         
	 PROB=INT(RVEC*(DOFS(1)+DOFS(2)+DOFS(3)+DOFS(4)))

	 DO J=1,12
	  IF (PROB .LT. J*6) THEN
	  HARDPARTICLE(I,1)=TABLEPARTICLE(J,2)
	  HARDPARTICLE(I,2)=TABLEPARTICLE(J,3)
	  HARDPARTICLE(I,3)=(-1)**(J+1)
          HARDMASS(I)=TABLEMASS(J)
	  GOTO 140
	  ENDIF
	 ENDDO
	 DO J=13,18
	  IF (PROB .LT. 72+(J-12)*2) THEN
	  HARDPARTICLE(I,1)=TABLEPARTICLE(J,2)
	  HARDPARTICLE(I,2)=TABLEPARTICLE(J,3)
	  HARDMASS(I)=TABLEMASS(J)
	  GOTO 140
	  ENDIF
	 ENDDO
	 DO J=19,24
	  IF (PROB .LT. 84+(J-18)*1) THEN
	  HARDPARTICLE(I,1)=TABLEPARTICLE(J,2)
	  GOTO 140
	  ENDIF
	 ENDDO	
          IF (PROB .LT. 106) THEN
	   HARDPARTICLE(I,1)=TABLEPARTICLE(25,2)
	   GOTO 140
	  ELSEIF (PROB .LT. 108) THEN
	   HARDPARTICLE(I,1)=TABLEPARTICLE(26,2)
	   GOTO 140
	  ENDIF
	  DO J=27,29
	   IF (PROB .LT. 108+(J-26)*3) THEN
	    HARDPARTICLE(I,1)=TABLEPARTICLE(J,2)
	    HARDPARTICLE(I,2)=TABLEPARTICLE(J,3)
	    HARDMASS(I)=TABLEMASS(J)	    	  
	    GOTO 140
	   ENDIF	  
	  ENDDO 
	   IF (PROB .LT. 117+INT(DOFS(4))) THEN
	    HARDPARTICLE(I,1)=TABLEPARTICLE(30,2)
	    GOTO 140
	   ELSE
	    HARDPARTICLE(I,1)=TABLEPARTICLE(31,2) 
	    HARDMASS(I)=TABLEMASS(31)	    	  
	   ENDIF
140        CONTINUE
	   ENDDO
	   
	   DO I=1,NP
	   THARDMASS=THARDMASS+HARDMASS(I)
           ENDDO
	   IF (THARDMASS .GT. QTHRESHOLD()) THEN
	   GOTO 10
	   ENDIF
	   
        EMCHARGE=0  

        DO I=1,100
          EMCHARGE=EMCHARGE+TABLEPARTICLE(I,1)*TABLEPARTICLE(I,3) 
        ENDDO
	
	DO I=1,NP
	  EMCHARGE=EMCHARGE+HARDPARTICLE(I,2)
	ENDDO 
		
      IF (IABS(EMCHARGE) .LE. 4) GOTO 200

        IF (EMCHARGEFLAG .EQV. .TRUE.) THEN 
	    GOTO 10
	ELSEIF (EMCHARGEFLAG .EQV. .FALSE.) THEN
	    CALL RANLUX(RVEC,4)
	    EMCHARGE=INT(5.*RVEC)
	    CALL RANLUX(RVEC,4)
	    IF (RVEC .LT. 0.5) THEN
	        EMCHARGE=-EMCHARGE
	    ENDIF
        ENDIF

200   DO I=1,100
	 NUMPARTICLE=NUMPARTICLE+TABLEPARTICLE(I,1)
      ENDDO      
     
      RETURN
      END SUBROUTINE FFRAGMENT
C====================================================================
C     SUBROUTINE PARTICLEMATRIX
C     Purpose: Gives the particle pythia codes, charges and masses
C====================================================================
	SUBROUTINE PARTICLEMATRIX(TABLEPARTICLE,TABLEMASS)

	IMPLICIT NONE
	INTEGER I,J,TABLEPARTICLE(100,3)
	DOUBLEPRECISION TABLEMASS(100),PYMASS
	
	EXTERNAL PYMASS
	
	
        DO I=1,100
	 DO J=1,3
	  TABLEPARTICLE(I,J)=0
	 ENDDO	
	  TABLEMASS(I)=0.0D0
	ENDDO
	
	TABLEPARTICLE(1,2)=1
	TABLEPARTICLE(2,2)=-1
	TABLEPARTICLE(3,2)=2
	TABLEPARTICLE(4,2)=-2
	TABLEPARTICLE(5,2)=3
	TABLEPARTICLE(6,2)=-3
	TABLEPARTICLE(7,2)=4
	TABLEPARTICLE(8,2)=-4
	TABLEPARTICLE(9,2)=5
	TABLEPARTICLE(10,2)=-5
	TABLEPARTICLE(11,2)=6
	TABLEPARTICLE(12,2)=-6
	TABLEPARTICLE(13,2)=11
	TABLEPARTICLE(14,2)=-11
	TABLEPARTICLE(15,2)=13
	TABLEPARTICLE(16,2)=-13
	TABLEPARTICLE(17,2)=15
	TABLEPARTICLE(18,2)=-15
	TABLEPARTICLE(19,2)=12
	TABLEPARTICLE(20,2)=-12
	TABLEPARTICLE(21,2)=14
	TABLEPARTICLE(22,2)=-14
	TABLEPARTICLE(23,2)=16
	TABLEPARTICLE(24,2)=-16
	TABLEPARTICLE(25,2)=21
	TABLEPARTICLE(26,2)=22
	TABLEPARTICLE(27,2)=23
	TABLEPARTICLE(28,2)=24
	TABLEPARTICLE(29,2)=-24
	TABLEPARTICLE(30,2)=39
	TABLEPARTICLE(31,2)=25
	
	TABLEPARTICLE(1,3)=-1
        TABLEPARTICLE(2,3)=1
        TABLEPARTICLE(3,3)=2
        TABLEPARTICLE(4,3)=-2
        TABLEPARTICLE(5,3)=-1
        TABLEPARTICLE(6,3)=1
        TABLEPARTICLE(7,3)=2
        TABLEPARTICLE(8,3)=-2
        TABLEPARTICLE(9,3)=-1
        TABLEPARTICLE(10,3)=1
        TABLEPARTICLE(11,3)=2
        TABLEPARTICLE(12,3)=-2
        TABLEPARTICLE(13,3)=-3
        TABLEPARTICLE(14,3)=3
        TABLEPARTICLE(15,3)=-3
        TABLEPARTICLE(16,3)=3
        TABLEPARTICLE(17,3)=-3
        TABLEPARTICLE(18,3)=3
        TABLEPARTICLE(28,3)=3
        TABLEPARTICLE(29,3)=-3
	        

	TABLEMASS(1)=PYMASS(1)*1.0D-3
        TABLEMASS(2)=PYMASS(-1)*1.0D-3
        TABLEMASS(3)=PYMASS(2)*1.0D-3
        TABLEMASS(4)=PYMASS(-2)*1.0D-3
        TABLEMASS(5)=PYMASS(3)*1.0D-3
        TABLEMASS(6)=PYMASS(-3)*1.0D-3
        TABLEMASS(7)=PYMASS(4)*1.0D-3
        TABLEMASS(8)=PYMASS(-4)*1.0D-3
        TABLEMASS(9)=PYMASS(5)*1.0D-3
        TABLEMASS(10)=PYMASS(-5)*1.0D-3
        TABLEMASS(11)=PYMASS(6)*1.0D-3
        TABLEMASS(12)=PYMASS(-6)*1.0D-3

        TABLEMASS(13)=PYMASS(11)*1.0D-3
        TABLEMASS(14)=PYMASS(-11)*1.0D-3
        TABLEMASS(15)=PYMASS(13)*1.0D-3
        TABLEMASS(16)=PYMASS(-13)*1.0D-3
        TABLEMASS(17)=PYMASS(15)*1.0D-3
        TABLEMASS(18)=PYMASS(-15)*1.0D-3
	TABLEMASS(27)=PYMASS(23)*1.0D-3
	TABLEMASS(28)=PYMASS(24)*1.0D-3
	TABLEMASS(29)=PYMASS(-24)*1.0D-3
        TABLEMASS(31)=PYMASS(25)*1.0D-3

	RETURN
	END SUBROUTINE PARTICLEMATRIX
C====================================================================
C     SUBROUTINE POLINTT
C     Purpose: Interpolates a function
C==================================================================== 
      SUBROUTINE POLINTT(XA,YA,N,X,Y,DY)
      IMPLICIT NONE

      INTEGER N,NMAX
      DOUBLEPRECISION DY,X,Y,XA(N),YA(N)
      PARAMETER (NMAX=10)
      INTEGER I,M,NS
      DOUBLEPRECISION DEN,DIF,DIFT,HO,HP,W,C(NMAX),D(NMAX)
      NS=1
      DIF=ABS(X-XA(1))
      DO 11 I=1,N
        DIFT=ABS(X-XA(I))
        IF (DIFT.LT.DIF) THEN
          NS=I
          DIF=DIFT
        ENDIF
        C(I)=YA(I)
        D(I)=YA(I)
11    CONTINUE
      Y=YA(NS)
      NS=NS-1
      DO 13 M=1,N-1
        DO 12 I=1,N-M
          HO=XA(I)-X
          HP=XA(I+M)-X
          W=C(I+1)-D(I)
          DEN=HO-HP
          IF(DEN.EQ.0.)PAUSE 'FAILURE IN POLINT'
          DEN=W/DEN
          D(I)=HP*DEN
          C(I)=HO*DEN
12      CONTINUE
        IF (2*NS.LT.N-M)THEN
          DY=C(NS+1)
        ELSE
          DY=D(NS)
          NS=NS-1
        ENDIF
        Y=Y+DY
13    CONTINUE
      RETURN
      END
C  (C) Copr. 1986-92 Numerical Recipes Software #>,1')5c).
C====================================================================
C     SUBROUTINE PJETS
C     Purpose: Computes the (random) four-momenta of the Hawking phase
C====================================================================  
      SUBROUTINE PJETS(S,B,MULTI,TABLEPARTICLE,TABLEMASS,ENERGY,
     ! PJET,PBH,CHECK)

      IMPLICIT NONE
      INTEGER I,J,K,MULTI,NI,MOMTRANSF
      INTEGER TABLEPARTICLE(100,3),CHECK
      DOUBLEPRECISION GAMMA,AL,BE,GAM,MBH,B
      DOUBLEPRECISION PJETF(3),PJETR(3), RK(1000),SUMRK,SUMRKN(3)
      DOUBLEPRECISION PBH(3,1000),PJET(3,1000),EN,TABLEMASS(100)
      DOUBLEPRECISION MST,ALPHA,S,XMIN,QMIN,ENERGY(1000),QTHRESHOLD
      DOUBLEPRECISION MASSPARTICLE(1000),MASSCHECK
      EXTERNAL MBH,QTHRESHOLD
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      REAL RVEC
      CHECK=1

      PJETF(1)=0.0D0
      PJETF(2)=0.0D0
      PJETF(3)=1.0D0
      
      SUMRK=0.0D0     

      DO J=1,3
	PJETR(J)=0.0D0
	SUMRKN(J)=0.0D0
	DO K=1,1000
          PBH(J,K)=0.0D0
	  PJET(J,K)=0.0D0
	ENDDO
      ENDDO

      DO J=1,MULTI       		
        CALL EULER(PJETF,PJETR)
	DO K=1,3
          PJET(K,J)=PJETR(K)
	ENDDO 
      ENDDO
      
      IF (MULTI .EQ. 1) THEN
        RK(1)=1.0D0
        EN=(MBH(B,S)**2.0D0-QTHRESHOLD()**2.0D0)/2.0D0/MBH(B,S)
        GOTO 100
      ELSE
        CONTINUE
      ENDIF
     
      DO J=1,1000
         
         CALL RANLUX(RVEC,4)
         RK(J)=1.0D0+(DBLE(RVEC)-0.5D0)/10.0D0

      ENDDO
      
      DO J=1,MULTI         
	 SUMRK=SUMRK+RK(J)
         DO K=1,3
	  SUMRKN(K)=SUMRKN(K)+RK(J)*PJET(K,J)
	 ENDDO
	ENDDO 

      AL=SUMRK**2.0D0-SUMRKN(1)**2.0D0-SUMRKN(2)**2.0D0-SUMRKN(3)**2.0D0
      BE=MBH(B,S)*SUMRK
      GAM=MBH(B,S)**2.0D0-QTHRESHOLD()**2.0D0
     
      EN=(BE-DSQRT(BE**2.0D0-AL*GAM))/AL
 
100     K=0
        DO I=1,100
          DO J=1,TABLEPARTICLE(I,1)
             MASSPARTICLE(K+J)=TABLEMASS(I)
	  ENDDO
	     K=K+TABLEPARTICLE(I,1)
        ENDDO
      
          DO J=1,MULTI         
          ENERGY(J)=EN*RK(J)
	  MASSCHECK = ENERGY(J)**2.0D0-MASSPARTICLE(J)**2.0D0
	  IF (MASSCHECK .LT. 0.0D0) THEN
	      CHECK=0
	      RETURN
	  ENDIF
          DO K=1,3
	   PJET(K,J)=DSQRT(MASSCHECK)*PJET(K,J)
	   PBH(K,J+1)=PBH(K,J)-PJET(K,J)
	  ENDDO
         ENDDO

      RETURN 
      END SUBROUTINE PJETS
C====================================================================  
C     SUBROUTINE EULER
C     Purpose: Randomly rotates a unit three-vector vector
C====================================================================  
      SUBROUTINE EULER(P,PR)

      IMPLICIT NONE
      DOUBLE PRECISION P(3),PR(3),PI
      DOUBLE PRECISION CTH,CPH,CPS,STH,SPH,SPS,TH,PH,PS
      COMMON/PARAMETERS/ PI

      REAL RVEC

      CALL RANLUX(RVEC,4)
      TH=DBLE(RVEC)*2.0D0*PI
      CALL RANLUX(RVEC,4)
      PH=DBLE(RVEC)*2.0D0*PI
      CALL RANLUX(RVEC,4)
      PS=DBLE(RVEC)*2.0D0*PI

      CTH=DCOS(TH)
      CPH=DCOS(PH)
      CPS=DCOS(PS)
      STH=DSIN(TH)
      SPH=DSIN(PH)
      SPS=DSIN(PS)
      
      PR(1)=P(1)*(CPS*CPH-CTH*SPH*SPS)+P(2)*(CPS*SPH+CTH*CPH*SPS)
     !   +P(3)*(SPS*STH)
      PR(2)=P(1)*(-SPS*CPH-CTH*SPH*CPS)+P(2)*(-SPS*SPH+CTH*CPH*CPS)
     !   +P(3)*(CPS*STH)
      PR(3)=P(1)*(STH*SPH)+P(2)*(-STH*CPH)+P(3)*(CTH)

      RETURN
      END SUBROUTINE EULER
C====================================================================
C     SUBROUTINE VECTORPARTICLE
C     Purpose: Returns the decay particles
C==================================================================== 
      SUBROUTINE VECTORPARTICLE(S,B,TABLEPARTICLE,HARDPARTICLE,
     ! NUMPARTICLE,VP,EMCHARGE)
      
      IMPLICIT NONE
      DOUBLEPRECISION VP(1000,6),PBH(3,1000),MBH,ENERGY(1000)
      DOUBLEPRECISION B,TENERGY,MST,ALPHA,S,XMIN,QMIN,PJET(3,1000)
      DOUBLEPRECISION PBPB,PBPK,EBH,QTHRESHOLD
      DOUBLEPRECISION TABLEMASS(100),HARDMASS(18),NF(4)
      INTEGER I,J,K,NUMPARTICLE,NP,NI,MOMTRANSF,CHECK
      INTEGER TABLEPARTICLE(100,3),HARDPARTICLE(18,3)
      INTEGER KGENEV,EMCHARGE
      REAL TECM,AMASS(18),PCM(5,18),WT

      EXTERNAL MBH,QTHRESHOLD
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/GENIN / NP, TECM, AMASS, KGENEV
      COMMON/GENOUT/ PCM, WT
 
10     CALL FFRAGMENT(S,B,NUMPARTICLE,TABLEPARTICLE,TABLEMASS,
     ! HARDPARTICLE,HARDMASS,EMCHARGE)
           
      K=0
      TENERGY=0.0D0
      
      CALL PJETS(S,B,NUMPARTICLE,TABLEPARTICLE,TABLEMASS,
     ! ENERGY,PJET,PBH,CHECK)

      IF (CHECK .EQ. 0) GOTO 10

      DO I=1,100
          DO J=1,TABLEPARTICLE(I,1)
	    VP(K+J,1)=TABLEPARTICLE(I,2)
	    VP(K+J,2)=ENERGY(K+J)
	    VP(K+J,3)=PJET(1,K+J)
	    VP(K+J,4)=PJET(2,K+J)
	    VP(K+J,5)=PJET(3,K+J)
	    VP(K+J,6)=DSQRT(PJET(1,K+J)**2.0D0+PJET(2,K+J)**2.0D0+
     !                PJET(3,K+J)**2.0D0)
	    TENERGY=TENERGY+ENERGY(K+J)
	  ENDDO
	  K=K+TABLEPARTICLE(I,1)
      ENDDO

      EBH=MBH(B,S)-TENERGY

      IF (NP .EQ. 0) THEN 
       VP(NUMPARTICLE+1,1)=80.0D0
       VP(NUMPARTICLE+1,2)=EBH
       VP(NUMPARTICLE+1,3)=PBH(1,NUMPARTICLE+1)
       VP(NUMPARTICLE+1,4)=PBH(2,NUMPARTICLE+1)
       VP(NUMPARTICLE+1,5)=PBH(3,NUMPARTICLE+1)
       VP(NUMPARTICLE+1,6)=DSQRT(PBH(1,NUMPARTICLE+1)**2.0D0+
     !     PBH(2,NUMPARTICLE+1)**2.0D0+PBH(3,NUMPARTICLE+1)**2.0D0)
      ELSE
        IF (NUMPARTICLE .EQ. 0) THEN
	   TECM=SNGL(EBH)
	ELSE
	   TECM=SNGL(QTHRESHOLD())
	ENDIF   	
       DO I=1,NP
	 AMASS(I)=SNGL(HARDMASS(I))
       ENDDO
       CALL GENBOD
       PBPB=PBH(1,NUMPARTICLE+1)**2.0D0+
     !    PBH(2,NUMPARTICLE+1)**2.0D0+ PBH(3,NUMPARTICLE+1)**2.0D0
       DO I=1,NP
          PBPK=PBH(1,NUMPARTICLE+1)*DREAL(PCM(1,I))+
     !    PBH(2,NUMPARTICLE+1)*DREAL(PCM(2,I))+
     !    PBH(3,NUMPARTICLE+1)*DREAL(PCM(3,I))
 
	  IF (NUMPARTICLE .EQ. 0) THEN
            VP(NUMPARTICLE+I,1)=HARDPARTICLE(I,1)
	    VP(NUMPARTICLE+I,2)=DREAL(PCM(4,I))
	    VP(NUMPARTICLE+I,3)=DREAL(PCM(1,I))
	    VP(NUMPARTICLE+I,4)=DREAL(PCM(2,I))
	    VP(NUMPARTICLE+I,5)=DREAL(PCM(3,I))
	    VP(NUMPARTICLE+I,6)=DREAL(PCM(5,I))
	  ELSE
          VP(NUMPARTICLE+I,1)=HARDPARTICLE(I,1)
	  VP(NUMPARTICLE+I,2)=(DREAL(PCM(4,I))*EBH+PBPK)/QTHRESHOLD()
	   VP(NUMPARTICLE+I,3)=DREAL(PCM(1,I))+PBH(1,NUMPARTICLE+1)
     !      /QTHRESHOLD()*(DREAL(PCM(4,I))+(EBH-QTHRESHOLD())*PBPK/PBPB)

	   VP(NUMPARTICLE+I,4)=DREAL(PCM(2,I))+PBH(2,NUMPARTICLE+1)
     !      /QTHRESHOLD()*(DREAL(PCM(4,I))+(EBH-QTHRESHOLD())*PBPK/PBPB)

	   VP(NUMPARTICLE+I,5)=DREAL(PCM(3,I))+PBH(3,NUMPARTICLE+1)
     !      /QTHRESHOLD()*(DREAL(PCM(4,I))+(EBH-QTHRESHOLD())*PBPK/PBPB)

	   VP(NUMPARTICLE+I,6)=DSQRT(VP(NUMPARTICLE+I,3)**2.0D0+
     !	   VP(NUMPARTICLE+I,4)**2.0D0+VP(NUMPARTICLE+I,5)**2.0D0)

          ENDIF
       ENDDO 
      ENDIF
      RETURN
      END SUBROUTINE VECTORPARTICLE
C====================================================================
C     SUBROUTINE DECAY
C     Purpose: Computes the total quantities of the decay
C==================================================================== 
      SUBROUTINE DECAY(NUMPARTICLE,VP,PROD_DECAY,P,DP,MP)

      IMPLICIT NONE
      INTEGER I,J,NI,PROD_DECAY(10),CODE,NUMPARTICLE,NP
      INTEGER MOMTRANSF
      DOUBLEPRECISION MST,ALPHA,XMIN,QMIN,VP(1000,6),P(10,5),MP(5),DP(5)
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      INTEGER KGENEV
      REAL TECM,AMASS(18),PCM(5,18),WT
 
      COMMON/GENIN / NP, TECM, AMASS, KGENEV
      COMMON/GENOUT/ PCM, WT
      
      
      DO I=1,10
         PROD_DECAY(I)=0
	 DO J=1,5
	    P(I,J)=0.0D0
	    DP(J)=0.0D0
	    MP(J)=0.0D0
	 ENDDO   
      ENDDO
      	
      IF (NP .EQ. 0) THEN
         DO I=1,NUMPARTICLE+1
	    CODE=IABS(INT(VP(I,1)))
	    IF (CODE .EQ. 0) THEN  
	        PROD_DECAY(1)=PROD_DECAY(1)+1
		DO J=1,5
		  P(1,J)=P(1,J)+VP(I,J+1)
		  MP(J)=MP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 25) THEN
	        PROD_DECAY(2)=PROD_DECAY(2)+1
		DO J=1,5
		  P(2,J)=P(2,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .LE. 6 .OR. CODE .EQ. 21) THEN 
	        PROD_DECAY(3)=PROD_DECAY(3)+1
		DO J=1,5
		  P(3,J)=P(3,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 11 .OR. CODE .EQ. 13 .OR. CODE .EQ. 15) THEN
	    	PROD_DECAY(4)=PROD_DECAY(4)+1
		DO J=1,5
		  P(4,J)=P(4,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 12 .OR. CODE .EQ. 14 .OR. CODE .EQ. 16) THEN	
	    	PROD_DECAY(5)=PROD_DECAY(5)+1
		DO J=1,5
		  P(5,J)=P(5,J)+VP(I,J+1)
		  MP(J)=MP(J)+VP(I,J+1)
		ENDDO
	    ELSEIF (CODE .EQ. 22) THEN	
	    	PROD_DECAY(6)=PROD_DECAY(6)+1
		DO J=1,5
		  P(6,J)=P(6,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 24) THEN	
	    	PROD_DECAY(7)=PROD_DECAY(7)+1
		DO J=1,5
		  P(7,J)=P(7,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 23) THEN	
	    	PROD_DECAY(8)=PROD_DECAY(8)+1
		DO J=1,5
		  P(8,J)=P(8,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 21) THEN	
	    	PROD_DECAY(9)=PROD_DECAY(9)+1
		DO J=1,5
		  P(9,J)=P(9,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 39) THEN	
	    	PROD_DECAY(10)=PROD_DECAY(10)+1
		DO J=1,5
		  P(10,J)=P(10,J)+VP(I,J+1)
		  MP(J)=MP(J)+VP(I,J+1)
		ENDDO  
	    ELSE
	        WRITE(99,*) 'CATASTROPHIC FAILURE IN SUBROUTINE DECAY: ',
     !               '[UNKNOWN PARTICLE CODE]'
	        STOP
	    ENDIF	
         ENDDO
       ELSE
	 DO I=1,NUMPARTICLE+NP
	    CODE=IABS(INT(VP(I,1)))
            IF (CODE .EQ. 0) THEN  
	    	PROD_DECAY(1)=PROD_DECAY(1)+1
		DO J=1,5
		  P(1,J)=P(1,J)+VP(I,J+1)
		  MP(J)=MP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 25) THEN
	        PROD_DECAY(2)=PROD_DECAY(2)+1
		DO J=1,5
		  P(2,J)=P(2,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .LE. 6 .OR. CODE .EQ. 21) THEN 
	        PROD_DECAY(3)=PROD_DECAY(3)+1
		DO J=1,5
		  P(3,J)=P(3,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 11 .OR. CODE .EQ. 13 .OR. CODE .EQ. 15) THEN
	    	PROD_DECAY(4)=PROD_DECAY(4)+1
		DO J=1,5
		  P(4,J)=P(4,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 12 .OR. CODE .EQ. 14 .OR. CODE .EQ. 16) THEN	
	    	PROD_DECAY(5)=PROD_DECAY(5)+1
		DO J=1,5
		  P(5,J)=P(5,J)+VP(I,J+1)
		  MP(J)=MP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 22) THEN	
	    	PROD_DECAY(6)=PROD_DECAY(6)+1
		DO J=1,5
		  P(6,J)=P(6,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 24) THEN	
	    	PROD_DECAY(7)=PROD_DECAY(7)+1
		DO J=1,5
		  P(7,J)=P(7,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 23) THEN	
	    	PROD_DECAY(8)=PROD_DECAY(8)+1
		DO J=1,5
		  P(8,J)=P(8,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 21) THEN	
	    	PROD_DECAY(9)=PROD_DECAY(9)+1
		DO J=1,5
		  P(9,J)=P(9,J)+VP(I,J+1)
		  DP(J)=DP(J)+VP(I,J+1)
		ENDDO  
	    ELSEIF (CODE .EQ. 39) THEN	
	    	PROD_DECAY(10)=PROD_DECAY(10)+1
		DO J=1,5
		  P(10,J)=P(10,J)+VP(I,J+1)
		  MP(J)=MP(J)+VP(I,J+1)
		ENDDO  
	    ELSE
	        WRITE(99,*) 'CATASTROPHIC FAILURE IN SUBROUTINE DECAY: ',
     !               '[UNKNOWN PARTICLE CODE]'
	        STOP
	    ENDIF
         ENDDO
      ENDIF     	
      
      RETURN         
      END SUBROUTINE DECAY
C====================================================================
C     SUBROUTINE RATIOS
C     Purpose: Computes the total quantities of the decay
C==================================================================== 
      SUBROUTINE RATIOS(PROD_DECAY,CL2HRATIO,PH2HRATIO)

      IMPLICIT NONE
      INTEGER PROD_DECAY(10),NHADRONS,NCLEPTONS,NPHOTONS
      DOUBLEPRECISION CL2HRATIO,PH2HRATIO
      COMMON/DECAY_PARAMETERS/NHADRONS,NCLEPTONS,NPHOTONS
      
      NHADRONS=NHADRONS+PROD_DECAY(3)+PROD_DECAY(9)
      NCLEPTONS=NCLEPTONS+PROD_DECAY(4)
      NPHOTONS=NPHOTONS+PROD_DECAY(6)
      
      IF (NHADRONS .GT. 0) THEN
      CL2HRATIO=DBLE(NCLEPTONS)/DBLE(NHADRONS)
      ELSE
      CL2HRATIO=-1
      ENDIF
      
      IF (NHADRONS .GT. 0) THEN
      PH2HRATIO=DBLE(NPHOTONS)/DBLE(NHADRONS)
      ELSE
      PH2HRATIO=-1
      ENDIF

      RETURN         
      END SUBROUTINE RATIOS
C====================================================================
C     SUBROUTINE BOOSTPTL
C     Purpose: Boosts a particle to the lab frame
C=======================================================================
      SUBROUTINE BOOSTPTL(PCM,PLAB,BOOST,UX,UY,UZ)

      IMPLICIT NONE

      DOUBLEPRECISION PCM(4),PLAB(4)
      DOUBLEPRECISION BOOST
      DOUBLEPRECISION BETA,UX,UY,UZ,TMP
      
      BETA=DSQRT(1.0D0-1.0D0/BOOST/BOOST)
      
      TMP = (UX*PCM(1)+UY*PCM(2)+UZ*PCM(3)) * BETA

      PLAB(1)=PCM(1)+BOOST*BETA*UX*(BOOST*TMP/(BOOST+1.0D0)-PCM(4))
      PLAB(2)=PCM(2)+BOOST*BETA*UY*(BOOST*TMP/(BOOST+1.0D0)-PCM(4))
      PLAB(3)=PCM(3)+BOOST*BETA*UZ*(BOOST*TMP/(BOOST+1.0D0)-PCM(4))
      PLAB(4)=BOOST*(PCM(4)-TMP)
        
      RETURN
      END SUBROUTINE BOOSTPTL
C====================================================================
C     SUBROUTINE MASSREDUCE(sred)
C     Purpose: Computes the final mass of the BH BEFORE 
C====================================================================
	SUBROUTINE MASSREDUCE(SRED)

	IMPLICIT NONE
	INTEGER NI,MOMTRANSF,YVER,XBINS,MRXBINS,UP,I
	DOUBLE PRECISION MST,ALPHA,XMIN,QMIN,MRED, LNX1,LNX2,RNDTMP
	DOUBLEPRECISION MMF,LNXMIN,LNXMAX,LNSTEPX,TCROSS,DICROSS
	DOUBLE PRECISION DLUMDLNX,MRDLUMDLNX,INTEGRATE,SRED,MBINS
	DOUBLE PRECISION YYN(100),ZYN(100),REALBMAX,FP(1),FTHRESHOLD
	DOUBLE PRECISION DICROSSTMP,ECM
	REAL RVEC
	LOGICAL YNFLAG,EMCHARGEFLAG
      
        COMMON/CODE_PARAMETERS/MBINS,XBINS,MRXBINS
	COMMON/YOSHINO/ZYN,YYN,UP,YVER		
        COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
	COMMON/IENERGY/ECM
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

	EXTERNAL DLUMDLNX, MRDLUMDLNX, INTEGRATE, REALBMAX,FTHRESHOLD

	FP(1)=0.0D0
	
	MMF 	= FTHRESHOLD()
	
	IF (YNFLAG .EQV. .TRUE.) THEN
	  LNXMIN = DLOG(MMF**2 / ECM / ECM /YYN(1)**2.0D0)
	ELSE	
	  LNXMIN = DLOG(MMF**2 / ECM / ECM) 
        ENDIF

	IF (DEXP(LNXMIN) .GE. 1.0D0) THEN
           WRITE(*,*) '\n ***************************************\n'
	   WRITE(*,*) 'NO BLACK HOLES CAN BE FORMED AT THIS CM ENERGY'
           WRITE(*,*) '\n ***************************************\n'
 	   STOP
	ENDIF

	LNSTEPX	= - LNXMIN / DBLE(MRXBINS)
	
	TCROSS	= INTEGRATE(DLUMDLNX,FP,1,LNXMIN,0.0D0,LNSTEPX,1.D-5)


	CALL RANLUX(RVEC,4)

	RNDTMP = DBLE(RVEC)
	DICROSS = 0.0D0
	MRED	= ECM


	LNX1	= LNXMIN
	LNX2	= LNX1 + LNSTEPX
	
	DO WHILE (LNX2 .LT. 0.0D0)
	
        DICROSSTMP = INTEGRATE(DLUMDLNX,FP,1,LNX1,LNX2,LNSTEPX,1.D-5)
	DICROSS = DICROSS + DICROSSTMP	

	IF (DICROSS/TCROSS .GT. RNDTMP) THEN
	MRED = INTEGRATE(MRDLUMDLNX,FP,1,LNX1,LNX2,LNSTEPX,1.D-5) / 
     !		   DICROSSTMP

	GOTO 20

	ENDIF
	
	LNX1	= LNX2
	LNX2	= LNX1 + LNSTEPX

	ENDDO

20	CONTINUE

	SRED = MRED**2.0d0

2000   FORMAT (13(1PE11.2))

	RETURN
	END
C====================================================================
C     FUNCTION DSIGDLNX()
C     Purpose: Computes differential cross section wrt ln(x)
C====================================================================
	DOUBLE PRECISION FUNCTION DSIGDLNX(LNX)
	IMPLICIT NONE
	DOUBLE PRECISION LNX, X,ECM,SREDUCE, MREDUCE,CROSS,INTPDF
	EXTERNAL CROSS,intpdf

        INTEGER NI,MOMTRANSF
	DOUBLE PRECISION MST,ALPHA,XMIN,QMIN
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
        COMMON/IENERGY/ECM
		
	X 	= DEXP(LNX)
	SREDUCE = X * ECM * ECM
	MREDUCE = DSQRT(SREDUCE)

	DSIGDLNX = X * CROSS(SREDUCE) * INTPDF(X)

	RETURN
	END 
C====================================================================
C     FUNCTION DLUMDLNX()
C     Purpose: Computes differential cross section wrt ln(x)
C====================================================================
	DOUBLE PRECISION FUNCTION DLUMDLNX(LNX)
	IMPLICIT NONE
	DOUBLE PRECISION LNX,X,ECM,INTPDF
	EXTERNAL INTPDF

        INTEGER NI,MOMTRANSF
	DOUBLE PRECISION MST,ALPHA,XMIN,QMIN
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
        COMMON/IENERGY/ECM
		
	X 	= DEXP(LNX)
	DLUMDLNX = X * INTPDF(X)

	RETURN
	END 
C====================================================================
C     FUNCTION MRDSIGDLNX()
C     Purpose: Computes differential cross section wrt ln(x)
C====================================================================
	DOUBLE PRECISION FUNCTION MRDSIGDLNX(LNX)

	IMPLICIT NONE
	DOUBLE PRECISION LNX, X,ECM,SREDUCE, MREDUCE,INTPDF,CROSS
	EXTERNAL CROSS,INTPDF

        INTEGER NI,MOMTRANSF
	DOUBLE PRECISION MST,ALPHA,XMIN,QMIN
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
        COMMON/IENERGY/ECM
	
	X 	= DEXP(LNX)
	SREDUCE = X * ECM * ECM
	MREDUCE = DSQRT(SREDUCE)
	  
	MRDSIGDLNX = MREDUCE * X * CROSS(SREDUCE) * INTPDF(X)

	RETURN
	END
C====================================================================
C     FUNCTION MRDLUMDLNX()
C     Purpose: Computes differential cross section wrt ln(x)
C====================================================================
	DOUBLE PRECISION FUNCTION MRDLUMDLNX(LNX)

	IMPLICIT NONE
	DOUBLE PRECISION LNX, X,ECM,SREDUCE, MREDUCE,INTPDF
	EXTERNAL INTPDF

        INTEGER NI,MOMTRANSF
	DOUBLE PRECISION MST,ALPHA,XMIN,QMIN
	COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
        COMMON/IENERGY/ECM
	
	X 	= DEXP(LNX)
	SREDUCE = X * ECM * ECM
	MREDUCE = DSQRT(SREDUCE)
	  
	MRDLUMDLNX = MREDUCE * X * INTPDF(X)

	RETURN
	END
C====================================================================
C      FUNCTION PDFT
C      Purpose: Evaluates the integrand of the pdfs (diff cr sect)
C====================================================================
       DOUBLEPRECISION FUNCTION PDFT(X)

       DOUBLE PRECISION X,T,Y,Q,FP(2)
       DOUBLE PRECISION PDFT1,PDFT2
       
       COMMON/INTPARAM/ FP
       
       CALL SETCTQ5(8)

       T=FP(1)
       Q=FP(2)
       
       Y=T/X
       PDFT1=CTQ5PDF(-5,Y,Q)
       PDFT1=PDFT1+CTQ5PDF(-4,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(-3,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(-2,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(-1,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(0,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(1,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(2,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(3,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(4,Y,Q)	   
       PDFT1=PDFT1+CTQ5PDF(5,Y,Q)

       PDFT2=CTQ5PDF(-5,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(-4,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(-3,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(-2,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(-1,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(0,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(1,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(2,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(3,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(4,X,Q)*PDFT1
       PDFT2=PDFT2+CTQ5PDF(5,X,Q)*PDFT1
       PDFT=PDFT2/X

       RETURN
       END FUNCTION PDFT  
C====================================================================
C      FUNCTION INTPDF
C      Purpose: Evaluates the integral of the pdfs (diff cr sect)
C====================================================================
      DOUBLEPRECISION FUNCTION INTPDF(T)

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF
      DOUBLE PRECISION MST,ALPHA,XMIN,QMIN,A,MBH,MOMQ,RSCHWA,M,D
      DOUBLE PRECISION T,LAMBDA,PDFT,INTEGRATE,FP(2),DXINIT,ECM

      EXTERNAL LAMBDA,PDFT,INTEGRATE

      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/IENERGY/ECM
      COMMON/INTPARAM/ FP
       
      MBH=SQRT(T*ECM*ECM)

      D=DBLE(NI+4)
      M=MBH/MST	
      RSCHWA = LAMBDA()*M**(1.0D0/(D-3.0D0))/MST	  	
	
      IF (MOMTRANSF .EQ. 1) THEN		

	  IF (MBH .LT. 10.D0) THEN
	      MOMQ = MBH * 1000.D0
	  ELSE
	      MOMQ = 10000.D0
	  ENDIF
	  	
      ELSEIF (MOMTRANSF .EQ. 2) THEN  	  

	  IF (1.D0/RSCHWA .LT. 10.D0) THEN
	      MOMQ = 1.D0/RSCHWA * 1000.D0
	  ELSE
	      MOMQ = 10000.D0
	  ENDIF

      ENDIF

      A=T
      FP(1)=T
      FP(2)=MOMQ
      DXINIT=(1.0D0-A)/1.0D2
	
      INTPDF=INTEGRATE(PDFT,FP,2,A,1.0D0,DXINIT,1.0D-5)
	
      RETURN
      END FUNCTION INTPDF   
C====================================================================
C      FUNCTION LHCCROSS
C      Purpose: Computes the total cross section
C====================================================================
      DOUBLEPRECISION FUNCTION LHCCROSS()

      IMPLICIT NONE
      DOUBLEPRECISION LHCCROSSY,LHCCROSSBD
      LOGICAL YNFLAG,EMCHARGEFLAG
      
      EXTERNAL LHCCROSSY,LHCCROSSBD

      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
        
      IF (YNFLAG .EQV. .TRUE.) THEN
        LHCCROSS=LHCCROSSY()
      ELSE
        LHCCROSS=LHCCROSSBD()
      ENDIF
      
      RETURN
      END FUNCTION LHCCROSS   
C====================================================================
C      FUNCTION INNERLHCCROSSY
C      Purpose: Computes the inner integral OF YN total cross section
C====================================================================
      DOUBLEPRECISION FUNCTION INNERLHCCROSSY(Z)

      IMPLICIT NONE
      INTEGER UP,YVER,I
      DOUBLEPRECISION FP(1),DXINIT,A,FTHRESHOLD,ARGLHCCR,ECM,ZZ
      DOUBLEPRECISION INTEGRATE,ZYN(100),YYN(100),Z,YYZ,DY,R0,BMAX

      EXTERNAL ARGLHCCR,FTHRESHOLD,INTEGRATE,R0,BMAX

      COMMON/IENERGY/ECM
      COMMON /YOSHINO/ ZYN,YYN,UP,YVER
      
      FP(1)=0.0D0

      IF (YVER .EQ. 1) THEN   
         ZZ=Z*BMAX(ECM*ECM)/R0(ECM*ECM)
      ENDIF                  
      	
      IF (ZYN(UP) .LT. ZZ) THEN
          CALL POLINTT(ZYN(UP-2),YYN(UP-2),3,ZZ,YYZ,DY)	     
	  GOTO 20
        ENDIF
        DO I=2,UP-1
	  IF (ZYN(I) .GT. ZZ) THEN
	    CALL POLINTT(ZYN(I-1),YYN(I-1),3,ZZ,YYZ,DY)
	    GOTO 20 
	  ENDIF
	ENDDO
            
20    A=(FTHRESHOLD())**2.0D0/ECM/ECM/YYZ**2.0D0

      IF (A .LT. 1.0D0) THEN 
          DXINIT=(1.0D0-A)/1.0D1
          INNERLHCCROSSY=2.0D0*Z*INTEGRATE(ARGLHCCR,FP,1,A,1.0D0,
     !        DXINIT,1.0D-2)
      ELSE
           INNERLHCCROSSY=0.0d0
      ENDIF
      
      RETURN
      END FUNCTION INNERLHCCROSSY   
C====================================================================
C      FUNCTION LHCCROSSY
C      Purpose: Computes the YN total cross section
C====================================================================
      DOUBLEPRECISION FUNCTION LHCCROSSY()

      IMPLICIT NONE
      DOUBLEPRECISION FP(1),DXINIT
      DOUBLEPRECISION INTEGRATE,INNERLHCCROSSY

      EXTERNAL INTEGRATE,INNERLHCCROSSY
            
      FP(1)=0.0D0
      DXINIT=1.0D-1

      LHCCROSSY=INTEGRATE(INNERLHCCROSSY,FP,1,0.0D0,1.0D0,
     !  DXINIT,1.0D-2)

      RETURN
      END FUNCTION LHCCROSSY   
C====================================================================
C      SUBROUTINE ARGLHCCR
C      Purpose: Computes the inner integral for the total cross section
C====================================================================
      DOUBLEPRECISION FUNCTION ARGLHCCR(T)

      IMPLICIT NONE
      DOUBLEPRECISION ECM,INTPDF,T,CROSS

      EXTERNAL INTPDF,CROSS
      COMMON/IENERGY/ECM
            
      ARGLHCCR=INTPDF(T)*CROSS(T*ECM*ECM)
      
      RETURN
      END FUNCTION ARGLHCCR
C====================================================================
C      SUBROUTINE LHCCROSSBD
C      Purpose: Computes the total cross section
C====================================================================
      DOUBLEPRECISION FUNCTION LHCCROSSBD()

      IMPLICIT NONE
      DOUBLEPRECISION FP(1),DXINIT,A,FTHRESHOLD,ARGLHCCR
      DOUBLEPRECISION INTEGRATE,ECM

      EXTERNAL ARGLHCCR,FTHRESHOLD,INTEGRATE
      COMMON/IENERGY/ECM
            
      FP(1)=0.0D0

      A=(FTHRESHOLD())**2/ECM/ECM	
      DXINIT=(1.0D0-A)/1.0D2

      LHCCROSSBD=INTEGRATE(ARGLHCCR,FP,1,A,1.0D0,DXINIT,1.0D-5)

      RETURN
      END FUNCTION LHCCROSSBD   
C====================================================================
C      SUBROUTINE PLOTLHCCROSS
C      Purpose: Plots the total cross section 
C====================================================================
      SUBROUTINE PLOTLHCCROSS()

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF,I,XBINS,MRXBINS
      DOUBLEPRECISION MST,ALPHA,XMIN,QMIN,ECM,LHCCROSS,LHC_CROSS
      DOUBLEPRECISION MINMASS,XMINSAVE,XMAX,XVALUE,MBINS
      
      COMMON/CODE_PARAMETERS/MBINS,XBINS,MRXBINS	
      COMMON/IENERGY/ECM
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF

      EXTERNAL LHCCROSS,MINMASS

      OPEN(UNIT=97,FILE='catfish-v2.10.crplot',STATUS='UNKNOWN')			     
 
      XMINSAVE=XMIN      
      XMAX=ECM/MST*0.75D0  ! min BH mass up to 3/4 of total ECM**2

      DO I=1,XBINS            
          LHC_CROSS=LHCCROSS()
	 IF (MINMASS() .GT. MST ) THEN 
            XVALUE=XMIN*MINMASS()
         ELSE 
            XVALUE=XMIN*MST
         ENDIF	 
         WRITE(97,2000) XVALUE, LHC_CROSS 
         XMIN=XMIN+(XMAX-XMINSAVE)/DBLE(XBINS)
      ENDDO

      CLOSE(97)

      XMIN=XMINSAVE

2000  FORMAT (2(1pe15.7))
      END SUBROUTINE PLOTLHCCROSS
C====================================================================
C      SUBROUTINE PLOTLHCDCROSS
C      Purpose: Plots the diff cross section in TeV units
C====================================================================
      SUBROUTINE PLOTLHCDCROSS()

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF,I,XBINS,MRXBINS,YVER,UP
      DOUBLEPRECISION MST,ALPHA,XMIN,QMIN,XM,DSIGDLNX,LHC_DCROSS
      DOUBLEPRECISION M,X,FTHRESHOLD ,XSTEP,MBINS,ECM   
      DOUBLEPRECISION ZYN(100),YYN(100)
      LOGICAL YNFLAG,EMCHARGEFLAG
      
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
      COMMON/YOSHINO/ ZYN,YYN,UP,YVER	
      COMMON/IENERGY/ECM
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/CODE_PARAMETERS/MBINS,XBINS,MRXBINS

      EXTERNAL DSIGDLNX,FTHRESHOLD

      OPEN(UNIT=95,FILE='catfish-v2.10.dcrplot',STATUS='UNKNOWN')			     
	
	IF (YNFLAG .EQV. .TRUE.) THEN
	  XM = FTHRESHOLD()**2 / ECM/ECM /YYN(1)**2.0D0
	ELSE	
	  XM = FTHRESHOLD()**2 / ECM/ECM 
        ENDIF

	IF (XM .GE. 1.0D0) THEN
           WRITE(*,*) '\n ***************************************\n'
	   WRITE(*,*) 'NO BLACK HOLES CAN BE FORMED AT THIS CM ENERGY'
           WRITE(*,*) '\n ***************************************\n'
 	   STOP
	ENDIF
	
      X=XM
      XSTEP= (1.0D0-XM) / DBLE(XBINS)

      DO I=1,XBINS  
    
         LHC_DCROSS=DSIGDLNX(DLOG(X))*2.0D0/DSQRT(X*ECM*ECM)     	 

         M=DSQRT(X*ECM*ECM)
         WRITE(95,2000) M, LHC_DCROSS 
	 X= XM + DBLE(I) * XSTEP         
         
      ENDDO

      CLOSE(95)

2000  FORMAT (2(1pe15.7))
      END SUBROUTINE PLOTLHCDCROSS
C====================================================================
C      SUBROUTINE PLOTLHCDNDM
C      Purpose: Plots dn/dM in tev units
C====================================================================
      SUBROUTINE PLOTLHCDNDM(LUM)

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF,I,YVER,UP,XBINS,MRXBINS
      DOUBLEPRECISION MST,ALPHA,XMIN,QMIN,XM,DLUMDLNX,LHC_DNDM
      DOUBLEPRECISION M,X,FTHRESHOLD,XSTEP,MBINS,LUM,ECM   
      DOUBLEPRECISION ZYN(100),YYN(100)
      LOGICAL YNFLAG,EMCHARGEFLAG
      
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
      COMMON/YOSHINO/ ZYN,YYN,UP,YVER	
      COMMON/IENERGY/ECM
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/CODE_PARAMETERS/MBINS,XBINS,MRXBINS

      EXTERNAL DLUMDLNX,FTHRESHOLD

      OPEN(UNIT=94,FILE='catfish-v2.10.dndmplot',STATUS='UNKNOWN')			     
	
	IF (YNFLAG .EQV. .TRUE.) THEN
	  XM = FTHRESHOLD()**2 / ECM/ECM /YYN(1)**2.0D0
	ELSE	
	  XM = FTHRESHOLD()**2 / ECM/ECM 
        ENDIF

	IF (XM .GE. 1.0D0) THEN
           WRITE(*,*) '\n ***************************************\n'
	   WRITE(*,*) 'NO BLACK HOLES CAN BE FORMED AT THIS CM ENERGY'
           WRITE(*,*) '\n ***************************************\n'
 	   STOP
	ENDIF
	
      X=XM

      DO WHILE (X .LT. 1.0D0)  
    
         LHC_DNDM=DLUMDLNX(DLOG(X))*2.0D0/DSQRT(X*ECM*ECM)*MBINS*LUM
     !          *19.733D0**2.0D0

         M=DSQRT(X*ECM*ECM)
         WRITE(94,2000) M, LHC_DNDM
	 X = X + MBINS*(2*DSQRT(X*ECM*ECM)+MBINS)/ECM/ECM        
         
      ENDDO

      CLOSE(94)

2000  FORMAT (2(1pe15.7))
      END SUBROUTINE PLOTLHCDNDM
C====================================================================
C      SUBROUTINE PLOTBHMASS
C      Purpose: Plots the black hole mass
C====================================================================
      SUBROUTINE PLOTBHMASS(BHARRAY)

      IMPLICIT NONE
      INTEGER NI,MOMTRANSF,BHARRAY(1000),I,IMAX
      DOUBLEPRECISION MST,ALPHA,XMIN,QMIN,ECM,M,B,S
	
      COMMON/IENERGY/ECM
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/BHMASS/M

      IMAX=INT(ECM*10.0D0)
      
      OPEN(UNIT=96,FILE='catfish-v2.10.massplot',STATUS='UNKNOWN')			           
      
      DO I=1,IMAX
         WRITE(96,2001) I/10.0D0, BHARRAY(I)
      ENDDO
            
      CLOSE(96)

2001  FORMAT (f15.7,I6)

      END SUBROUTINE PLOTBHMASS
C====================================================================
C     FUNCTION INTEGRATE
C====================================================================
      FUNCTION INTEGRATE(FUNC,FP,NP,A,B,DXINIT,EPS)
      
      IMPLICIT NONE

      INTEGER NP,MAXSTEPS,NSTEP
      DOUBLE PRECISION INTEGRATE
      DOUBLE PRECISION FUNC      
      DOUBLE PRECISION A, B, EPS, DXINIT, FP(NP)
      DOUBLE PRECISION X, DX, DXNEXT, Y, DYDX, YSCALE
      
      EXTERNAL FUNC
      
      PARAMETER(MAXSTEPS=10000000)

      X     = A
      DX    = DXINIT
      Y     = 0.D0
      NSTEP = 0

      DO WHILE ((X-B)*(B-A).LT.0.D0.AND.NSTEP.LT.MAXSTEPS)

        NSTEP = NSTEP + 1
        DYDX = FUNC(X,FP,NP)


        YSCALE = MAX(DABS(Y) + DABS(DX*DYDX), 1.D-12)

        IF ((X+DX-B)*(X+DX-A).GT.0.D0)  ! IF STEPSIZE OVERSHOOTS, DECREASE IT.
     !    DX = B - X

        CALL RUNGE5VAR(Y,DYDX,X,DX,EPS,YSCALE,DXNEXT,FUNC,FP,NP)
        DX = DXNEXT

      ENDDO

      IF (NSTEP.GE.MAXSTEPS) THEN
        OPEN(UNIT=98,FILE='catfish-v2.10.warning',STATUS='UNKNOWN')			
        WRITE (98,*) 'Warning: Failed to converge in INTEGRATE.'
	CLOSE(98)
      ENDIF
      
      INTEGRATE = Y

      RETURN
      END
C====================================================================
C      SUBROUTINE RUNGE5VAR
C====================================================================
      SUBROUTINE RUNGE5VAR(Y,DYDX,X,HTRY,EPS,YSCALE,HNEXT,DERIVS,FP,NP)

      IMPLICIT NONE

      INTEGER NP 
      DOUBLE PRECISION FP(NP)
      DOUBLE PRECISION EPS,HNEXT,HTRY,X,DYDX,Y,YSCALE,DERIVS      
      DOUBLE PRECISION ERRMAX,H,HOLD,HTEMP,XNEW,YERR,YTEMP
      DOUBLE PRECISION SAFETY,PGROW,PSHRINK,ERRCON

      EXTERNAL DERIVS

      PARAMETER (SAFETY  =  0.9D0)
      PARAMETER (PGROW   = -0.2D0)
      PARAMETER (PSHRINK = -0.25D0)
      PARAMETER (ERRCON  =  1.89D-4)

      YERR = 0.D0
      H = HTRY                   ! Set stepsize to initial accuracy.
      ERRMAX = 10.D0

      DO WHILE (ERRMAX.GT.1.D0)

         CALL RUNGE(Y,DYDX,X,H,YTEMP,YERR,DERIVS,FP,NP)

         ERRMAX = DABS(YERR/YSCALE)/EPS 

         IF (ERRMAX.GT.1.D0) THEN ! Truncation error too large; reduce h
            
          HTEMP = SAFETY*H*(ERRMAX**PSHRINK)
          HOLD = H
          
          H = SIGN(MAX(DABS(HTEMP),0.1D0*DABS(H)),H) 

          XNEW = X + H
          IF (XNEW.EQ.X) THEN
            OPEN(UNIT=98,FILE='catfish-v2.10.warning',STATUS='UNKNOWN')			
            WRITE (98,*) 'Warning: Stepsize underflow in RUNGE5VAR().'
	    CLOSE(98)
            H = HOLD
            ERRMAX = 0.D0
          ENDIF
        ENDIF
      ENDDO

      IF (ERRMAX.GT.ERRCON) THEN
        HNEXT = SAFETY*H*(ERRMAX**PGROW)
      ELSE
        HNEXT = 5.D0 * H                
      ENDIF

      X = X + H
      Y = YTEMP

      RETURN
      END
C====================================================================
C      SUBROUTINE RUNGE
C====================================================================
      SUBROUTINE RUNGE(y,dydx,x,h,yout,yerr,DERIVS,fp,np)
     
      IMPLICIT NONE

      INTEGER NP
      DOUBLE PRECISION H,X,DYDX,Y,YERR,YOUT,DERIVS,FP(NP)
      DOUBLE PRECISION AK3, AK4, AK5 ,AK6
      DOUBLE PRECISION A2,A3,A4,A5,A6
      DOUBLE PRECISION C1,C3,C4,C6,DC1,DC3,DC4,DC5,DC6

      EXTERNAL DERIVS

      PARAMETER(A2  =    0.2D0)
      PARAMETER(A3  =    0.3D0)
      PARAMETER(A4  =    0.6D0)
      PARAMETER(A5  =    1.D0)
      PARAMETER(A6  =    0.875D0)
      PARAMETER(C1  =   37.D0/378.D0)
      PARAMETER(C3  =  250.D0/621.D0)
      PARAMETER(C4  =  125.D0/594.D0)
      PARAMETER(C6  =  512.D0/1771.D0)
      PARAMETER(DC1 = C1 -  2825.D0/27648.D0)
      PARAMETER(DC3 = C3 - 18575.D0/48384.D0)
      PARAMETER(DC4 = C4 - 13525.D0/55296.D0)
      PARAMETER(DC5 = -277.D0/14336.D0)
      PARAMETER(DC6 = C6 -     0.25D0)

      AK3 = DERIVS(X+A3*H,FP,NP)
      AK4 = DERIVS(X+A4*H,FP,NP)
      AK5 = DERIVS(X+A5*H,FP,NP)
      AK6 = DERIVS(X+A6*H,FP,NP)

      YOUT=Y+H*(C1*DYDX+C3*AK3+C4*AK4+C6*AK6)

      YERR=H*(DC1*DYDX+DC3*AK3+DC4*AK4+DC5*AK5+DC6*AK6)

      RETURN
      END
C====================================================================
C	SUBROUTINE READGLOBAL
C	Purpose: Reads the BH parameters
C=======================================================================
      SUBROUTINE READGLOBAL

      IMPLICIT NONE

      CHARACTER*80 LINESTRING,LINETEMP,LINETEMP2

      INTEGER I,NI,MOMTRANSF,YVER,NP,UP,KGENEV,RSEED,MLIST
      INTEGER YNFLAGI,EMCHARGEFLAGI,NUMEVENTS
      REAL TECM,AMASS(18)
      DOUBLE PRECISION MST,ALPHA,XMIN,QMIN,ZYN(100),YYN(100),CUTS(6)
      LOGICAL YNFLAG,EMCHARGEFLAG      

      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
      COMMON/YOSHINO/ZYN,YYN,UP,YVER
      COMMON/GENIN/NP, TECM, AMASS, KGENEV
      COMMON/SIMULATION/CUTS,NUMEVENTS,RSEED,MLIST

     
      OPEN(UNIT=12,FILE='catfish-v2.10.inp',STATUS='UNKNOWN',ERR=250)		
      OPEN(UNIT=13,FILE='catfish-v2.10.inp',STATUS='UNKNOWN',ERR=250)		
      

100      READ(12, '(A)',END=200) LINESTRING
         IF (LINESTRING(1:5) .EQ. 'MSTAR') THEN
             READ(13, *,ERR=300) LINETEMP, MST, LINETEMP2
	     IF (MST .LE. 0.0D0 .OR. MST .GT. 14.0D0) GOTO 350
             IF (LINETEMP2 .EQ. 'GeV') THEN
	     MST=MST/1000.	     
	     GOTO 100
	     ELSEIF (LINETEMP2 .NE. 'TeV') THEN
	     GOTO 400
	     ENDIF
	     GOTO 100
	 ELSEIF (LINESTRING(1:9) .EQ. 'NEXTRADIM') THEN
	     READ(13, *,ERR=500) LINETEMP, NI
	     IF (NI .LT. 3 .OR. NI .GT. 7) GOTO 600
	     GOTO 100
	 ELSEIF (LINESTRING(1:12) .EQ. 'GRAVITONLOSS') THEN
             READ(13, *,ERR=700) LINETEMP, YNFLAGI
             IF (YNFLAGI .EQ. 0) THEN
	      YNFLAG=.FALSE.
	     ELSEIF (YNFLAGI .EQ. 1) THEN
	      YNFLAG=.TRUE.
	     ELSE
	      GOTO 700
	     ENDIF
	     GOTO 100 
	 ELSEIF (LINESTRING(1:13) .EQ. 'GRAVITONMODEL') THEN
             READ(13, *,ERR=800) LINETEMP, YVER
             IF (YVER .NE. 0 .AND. YVER .NE. 1) GOTO 800
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:4) .EQ. 'XMIN') THEN
             READ(13, *,ERR=900) LINETEMP, XMIN
             IF (XMIN .LT. 1.0D0) GOTO 900
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:4) .EQ. 'QMIN') THEN
             READ(13, *,ERR=1000) LINETEMP, QMIN
             IF (QMIN .LT. 1.0D0) GOTO 1000
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:2) .EQ. 'NP') THEN
             READ(13, *,ERR=1100) LINETEMP, NP 
             IF (QMIN .LT. 0.0D0 .OR. QMIN .GT. 18) GOTO 1100
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:3) .EQ. 'MTR') THEN
             READ(13, *,ERR=1200) LINETEMP, MOMTRANSF
             IF (MOMTRANSF .NE. 1 .AND. MOMTRANSF .NE. 2) GOTO 1200	                   
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:8) .EQ. 'EMCHARGE') THEN
             READ(13, *,ERR=1300) LINETEMP, EMCHARGEFLAGI
             IF (EMCHARGEFLAGI .EQ. 0) THEN
              EMCHARGEFLAG=.FALSE.
	     ELSEIF (EMCHARGEFLAGI .EQ. 1) THEN
	      EMCHARGEFLAG=.TRUE.
	     ELSE
	      GOTO 1300
	     ENDIF 
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:5) .EQ. 'ALPHA') THEN
             READ(13, *,ERR=1400) LINETEMP, ALPHA      
             IF (ALPHA .LT. 0.0D0) GOTO 1400	                   
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:9) .EQ. 'NUMEVENTS') THEN
             READ(13, *,ERR=1500) LINETEMP, NUMEVENTS      
             IF (NUMEVENTS .LE. 0) GOTO 1500	                   
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:5) .EQ. 'RSEED') THEN
             READ(13, *,ERR=1600) LINETEMP, RSEED      
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:5) .EQ. 'MLIST') THEN
             READ(13, *,ERR=1700) LINETEMP, MLIST      
	     IF (MLIST .NE. 1 .AND. MLIST .NE. 2 .AND. MLIST .NE. 3
     !           .AND. MLIST .NE. 5 .AND. MLIST .NE. 7 .AND. MLIST
     !           .NE. 11 .AND. MLIST .NE. 12 .AND. MLIST .NE. 13)
     !           GOTO 1700
	     GOTO 100	            
	 ELSEIF (LINESTRING(1:4) .EQ. 'CUTS') THEN
             READ(13, *,ERR=1800) LINETEMP, CUTS(1), CUTS(2)      
	     GOTO 100
	 ELSEIF (LINESTRING(1:3) .EQ. 'PTL') THEN
             READ(13, *,ERR=1900) LINETEMP, CUTS(3)      
	     GOTO 100
	 ELSEIF (LINESTRING(1:4) .EQ. 'PRAP') THEN
             READ(13, *,ERR=2000) LINETEMP, CUTS(4)      
	     GOTO 100
	 ELSEIF (LINESTRING(1:5) .EQ. 'PTISO') THEN
             READ(13, *,ERR=2100) LINETEMP, CUTS(5)      
	     GOTO 100
	 ELSEIF (LINESTRING(1:4) .EQ. 'CONE') THEN
             READ(13, *,ERR=2200) LINETEMP, CUTS(6)      
	     GOTO 100            	            
         ELSE
	 READ(13, '(A)',END=200) LINESTRING
         GOTO 100
	 ENDIF


200   CLOSE(12)
220   CLOSE(13)


      IF (ALPHA .GT. 10.0D0) THEN 	
          OPEN(UNIT=98,FILE='catfish-v2.10.warning',STATUS='UNKNOWN')			
          WRITE(98,*) 'Warning in start-up checks: ',
     !               '[Parameter alpha is expected to be of order one]'
          CLOSE(98)
      ELSEIF (EMCHARGEFLAG .EQV. .FALSE.) THEN 	
           OPEN(UNIT=98,FILE='catfish-v2.10.warning',STATUS='UNKNOWN')			
           WRITE(98,*) 'Warning in start-up checks: ',
     !               '[No EM charge conservation in BH decay]'
          CLOSE(98)
      ENDIF

      IF (YNFLAG .EQV. .TRUE.) CALL YN()
      
      RETURN

250   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Unreadable or missing file catfish-v2.10.inp'
      CLOSE(99)
      STOP      


300   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [Parameter does not match its definition]'
      CLOSE(99)
      STOP      

350   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [MSTAR is unphysical or larger than LHC c.o.m. energy]'
      CLOSE(99)
      STOP      

400   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [MSTAR units]' 
      CLOSE(99)
      STOP      

500   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [NEXTRADIM is not integer]' 
      CLOSE(99)
      STOP      

600   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [NEXTRADIM < 3 or > 7]' 
      CLOSE(99)
      STOP      

700   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [GRAVITONLOSS]' 
      CLOSE(99)
      STOP      


800   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [GRAVITONMODEL]' 
      CLOSE(99)
      STOP      

900   OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [MINBHMASS is below the fundamental scale]' 
      CLOSE(99)
      STOP      

1000  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [MINEVAPMASS is below the fundamental scale]' 
      CLOSE(99)
      STOP      

1100  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [FINALDECAY]' 
      CLOSE(99)
      STOP      

1200  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [MOMTRANSF]' 
      CLOSE(99)
      STOP      

1300  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [EMCHARGE]' 
      CLOSE(99)
      STOP      

1400  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [ALPHA]' 
      CLOSE(99)
      STOP      

1500  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [NUMEVENTS]' 
      CLOSE(99)
      STOP      

1600  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [RSEED]' 
      CLOSE(99)
      STOP      

1700  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [MLIST]' 
      CLOSE(99)
      STOP      

1800  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [CUTS]' 
      CLOSE(99)
      STOP   
         
1900  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [PTL]' 
      CLOSE(99)
      STOP  
      
2000  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [PRAP]' 
      CLOSE(99)
      STOP        

2100  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [PTISO]' 
      CLOSE(99)
      STOP  
      
2200  OPEN(UNIT=99,FILE='catfish-v2.10.error',STATUS='UNKNOWN')		
      WRITE(99,*) 'Incorrect input of parameters in file catfish-v2.10.inp:
     ! [CONE]' 
      CLOSE(99)
      STOP        
      
      END SUBROUTINE READGLOBAL
C====================================================================  
C     SUBROUTINE GENBOD
C====================================================================
      SUBROUTINE GENBOD
      COMMON/GENIN / NP, TECM, AMASS(18), KGENEV
      COMMON/GENOUT/ PCM(5,18) , WT
      DIMENSION EMM(18)
      DIMENSION RNO(50)
      DIMENSION EM(18),PD(18),EMS(18),SM(18),FFQ(18),PCM1(90)
      EQUIVALENCE (NT,NP),(AMASS(1),EM(1)),(PCM1(1),PCM(1,1))
      DATA FFQ/0.,3.141592, 19.73921, 62.01255, 129.8788, 204.0131,
     2                       256.3704, 268.4705, 240.9780, 189.2637,
     3                       132.1308,  83.0202,  47.4210,  24.8295,
     4                        12.0006,   5.3858,   2.2560,   0.8859/
      DATA KNT,TWOPI/0,6.2831853073/
      KNT=KNT + 1
      IF(KNT.GT.1) GOTO 100
  100 CONTINUE
      IF(NT.LT.2) GOTO 1001
      IF(NT.GT.18) GOTO 1002
      NTM1=NT-1
      NTM2=NT-2
      NTP1=NT+1
      NTNM4=3*NT - 4
      EMM(1)=EM(1)
      TM=0.0
      DO 200 I=1,NT
      EMS(I)=EM(I)**2
      TM=TM+EM(I)
 200  SM(I)=TM
      TECMTM=TECM-TM
      IF(TECMTM.LE.0.0) GOTO 1000
      EMM(NT)=TECM
      IF(KGENEV.GT.1) GOTO 400
      EMMAX=TECMTM+EM(1)
      EMMIN=0.0
      WTMAX=1.0
      DO 350 I=2,NT
      EMMIN=EMMIN+EM(I-1)
      EMMAX=EMMAX+EM(I)
  350 WTMAX=WTMAX*PDK(EMMAX,EMMIN,EM(I))
      WTMAXQ=1.0/WTMAX
      GOTO 455
  400 WTMAXQ=TECMTM**NTM2*FFQ(NT) / TECM
  455 CONTINUE
      DO 457 I= 1, NTNM4
  457 RNO(I)=RNDM(I)
      IF(NTM2) 900,509,460
  460 CONTINUE
      CALL FLPSOR(RNO,NTM2)
      DO 508 J=2,NTM1
  508 EMM(J)=RNO(J-1)*(TECMTM)+SM(J)
  509 WT=WTMAXQ
      IR=NTM2
      DO 530 I=1,NTM1
      PD(I)=PDK(EMM(I+1),EMM(I),EM(I+1))
  530 WT=WT*PD(I)
      PCM(1,1)=0.0
      PCM(2,1)=PD(1)
      PCM(3,1)=0.0
      DO 570 I=2,NT
      PCM(1,I)=0.0
      PCM(2,I)=-PD(I-1)
      PCM(3,I)=0.0
      IR=IR+1
      BANG=TWOPI*RNO(IR)
      CB=COS(BANG)
      SB=SIN(BANG)
      IR=IR+1
      C=2.0*RNO(IR)-1.0
      S=SQRT(1.0-C*C)
      IF(I.EQ.NT) GOTO 1567
      ESYS=SQRT(PD(I)**2+EMM(I)**2)
      BETA=PD(I)/ESYS
      GAMA=ESYS/EMM(I)
      DO 568 J=1,I
      NDX=5*J - 5
      AA= PCM1(NDX+1)**2 + PCM1(NDX+2)**2 + PCM1(NDX+3)**2
      PCM1(NDX+5)=SQRT(AA)
      PCM1(NDX+4)=SQRT(AA+EMS(J))
      CALL ROTES2(C,S,CB,SB,PCM,J)
      PSAVE=GAMA*(PCM(2,J)+BETA*PCM(4,J))
  568 PCM(2,J)=PSAVE
      GOTO 570
 1567 DO 1568 J=1,I
      AA=PCM(1,J)**2 + PCM(2,J)**2 + PCM(3,J)**2
      PCM(5,J)=SQRT(AA)
      PCM(4,J)=SQRT(AA+EMS(J))
      CALL ROTES2(C,S,CB,SB,PCM,J)
 1568 CONTINUE
  570 CONTINUE
  900 CONTINUE
      RETURN
 1000 WRITE(6,1100)
      GOTO 1050
 1001 WRITE(6,1101)
      GOTO 1050
 1002 WRITE(6,1102)
 1050 WRITE(6,1150) KNT
      WRITE(6,1200) NP,TECM,(AMASS(JK),JK=1,NP)
      STOP
 1100 FORMAT(28H0 AVAILABLE ENERGY NEGATIVE )
 1101 FORMAT(33H0 LESS THAN 2 OUTGOING PARTICLES )
 1102 FORMAT(34H0 MORE THAN 18 OUTGOING PARTICLES )
 1150 FORMAT(47H0 ABOVE ERROR DETECTED IN GENBOD AT CALL NUMBER,I7)
 1160 FORMAT(34H0 FIRST CALL TO SUBROUTINE GENBOD )
 1200 FORMAT(36H  INPUT DATA TO GENBOD.         NP=   ,I6/
     +  ,8H   TECM=,E16.7,18H  PARTICLE MASSES=,5E15.5/(42X,5E15.5)
     +)
      END
c============================================================================
c     SUBROUTINE RANLUX
c============================================================================
      SUBROUTINE RANLUX(RVEC,LENV)
      DIMENSION RVEC(LENV)
      DIMENSION SEEDS(24), ISEEDS(24), ISDEXT(25)
      PARAMETER (MAXLEV=4, LXDFLT=3)
      DIMENSION NDSKIP(0:MAXLEV)
      DIMENSION NEXT(24)
      PARAMETER (TWOP12=4096., IGIGA=1000000000,JSDFLT=314159265)
      PARAMETER (ITWO24=2**24, ICONS=2147483563)
      SAVE NOTYET, I24, J24, CARRY, SEEDS, TWOM24, TWOM12, LUXLEV
      SAVE NSKIP, NDSKIP, IN24, NEXT, KOUNT, MKOUNT, INSEED
      INTEGER LUXLEV
      LOGICAL NOTYET
      DATA NOTYET, LUXLEV, IN24, KOUNT, MKOUNT /.TRUE., LXDFLT, 0,0,0/
      DATA I24,J24,CARRY/24,10,0./
      DATA NDSKIP/0,   24,   73,  199,  365 /
      IF (NOTYET) THEN
         NOTYET = .FALSE.
         JSEED = JSDFLT  
         INSEED = JSEED
         WRITE(6,'(A,I12)') ' RANLUX DEFAULT INITIALIZATION: ',JSEED
         LUXLEV = LXDFLT
         NSKIP = NDSKIP(LUXLEV)
         LP = NSKIP + 24
         IN24 = 0
         KOUNT = 0
         MKOUNT = 0
         WRITE(6,'(A,I2,A,I4)')  ' RANLUX DEFAULT LUXURY LEVEL =  ',
     +        LUXLEV,'      p =',LP
            TWOM24 = 1.
         DO 25 I= 1, 24
            TWOM24 = TWOM24 * 0.5
         K = JSEED/53668
         JSEED = 40014*(JSEED-K*53668) -K*12211
         IF (JSEED .LT. 0)  JSEED = JSEED+ICONS
         ISEEDS(I) = MOD(JSEED,ITWO24)
   25    CONTINUE
         TWOM12 = TWOM24 * 4096.
         DO 50 I= 1,24
         SEEDS(I) = REAL(ISEEDS(I))*TWOM24
         NEXT(I) = I-1
   50    CONTINUE
         NEXT(1) = 24
         I24 = 24
         J24 = 10
         CARRY = 0.
         IF (SEEDS(24) .EQ. 0.) CARRY = TWOM24
      ENDIF
      DO 100 IVEC= 1, LENV
      UNI = SEEDS(J24) - SEEDS(I24) - CARRY 
      IF (UNI .LT. 0.)  THEN
         UNI = UNI + 1.0
         CARRY = TWOM24
      ELSE
         CARRY = 0.
      ENDIF
      SEEDS(I24) = UNI
      I24 = NEXT(I24)
      J24 = NEXT(J24)
      RVEC(IVEC) = UNI
      IF (UNI .LT. TWOM12)  THEN
         RVEC(IVEC) = RVEC(IVEC) + TWOM24*SEEDS(J24)
         IF (RVEC(IVEC) .EQ. 0.)  RVEC(IVEC) = TWOM24*TWOM24
      ENDIF
      IN24 = IN24 + 1
      IF (IN24 .EQ. 24)  THEN
         IN24 = 0
         KOUNT = KOUNT + NSKIP
         DO 90 ISK= 1, NSKIP
         UNI = SEEDS(J24) - SEEDS(I24) - CARRY
         IF (UNI .LT. 0.)  THEN
            UNI = UNI + 1.0
            CARRY = TWOM24
         ELSE
            CARRY = 0.
         ENDIF
         SEEDS(I24) = UNI
         I24 = NEXT(I24)
         J24 = NEXT(J24)
   90    CONTINUE
      ENDIF
  100 CONTINUE
      KOUNT = KOUNT + LENV
      IF (KOUNT .GE. IGIGA)  THEN
         MKOUNT = MKOUNT + 1
         KOUNT = KOUNT - IGIGA
      ENDIF
      RETURN
      ENTRY RLUXIN(ISDEXT)
*     IF block added by Phillip Helbig after correpondence with James
      IF (NOTYET) THEN
         WRITE(6,'(A)')  ' PROPER RESULTS ONLY WITH INITIALISATION FROM 
     $25 INTEGERS OBTAINED WITH RLUXUT'
         NOTYET = .FALSE.
      ENDIF
         TWOM24 = 1.
         DO 195 I= 1, 24
         NEXT(I) = I-1
  195    TWOM24 = TWOM24 * 0.5
         NEXT(1) = 24
         TWOM12 = TWOM24 * 4096.
      WRITE(6,'(A)') ' FULL INITIALIZATION OF RANLUX WITH 25 INTEGERS:'
      WRITE(6,'(5X,5I12)') ISDEXT
      DO 200 I= 1, 24
      SEEDS(I) = REAL(ISDEXT(I))*TWOM24
  200 CONTINUE
      CARRY = 0.
      IF (ISDEXT(25) .LT. 0)  CARRY = TWOM24
      ISD = IABS(ISDEXT(25))
      I24 = MOD(ISD,100)
      ISD = ISD/100
      J24 = MOD(ISD,100)
      ISD = ISD/100
      IN24 = MOD(ISD,100)
      ISD = ISD/100
      LUXLEV = ISD
        IF (LUXLEV .LE. MAXLEV) THEN
          NSKIP = NDSKIP(LUXLEV)
          WRITE (6,'(A,I2)') ' RANLUX LUXURY LEVEL SET BY RLUXIN TO: ',
     +                         LUXLEV
        ELSE  IF (LUXLEV .GE. 24) THEN
          NSKIP = LUXLEV - 24
          WRITE (6,'(A,I5)') ' RANLUX P-VALUE SET BY RLUXIN TO:',LUXLEV
        ELSE
          NSKIP = NDSKIP(MAXLEV)
          WRITE (6,'(A,I5)') ' RANLUX ILLEGAL LUXURY RLUXIN: ',LUXLEV
          LUXLEV = MAXLEV
        ENDIF
      INSEED = -1
      RETURN
      ENTRY RLUXUT(ISDEXT)
      DO 300 I= 1, 24
         ISDEXT(I) = INT(SEEDS(I)*TWOP12*TWOP12)
  300 CONTINUE
      ISDEXT(25) = I24 + 100*J24 + 10000*IN24 + 1000000*LUXLEV
      IF (CARRY .GT. 0.)  ISDEXT(25) = -ISDEXT(25)
      RETURN
      ENTRY RLUXAT(LOUT,INOUT,K1,K2)
      LOUT = LUXLEV
      INOUT = INSEED
      K1 = KOUNT
      K2 = MKOUNT
      RETURN
      ENTRY RLUXGO(LUX,INS,K1,K2)
         IF (LUX .LT. 0) THEN
            LUXLEV = LXDFLT
         ELSE IF (LUX .LE. MAXLEV) THEN
            LUXLEV = LUX
         ELSE IF (LUX .LT. 24 .OR. LUX .GT. 2000) THEN
            LUXLEV = MAXLEV
            WRITE (6,'(A,I7)') ' RANLUX ILLEGAL LUXURY RLUXGO: ',LUX
         ELSE
            LUXLEV = LUX
            DO 310 ILX= 0, MAXLEV
              IF (LUX .EQ. NDSKIP(ILX)+24)  LUXLEV = ILX
  310       CONTINUE
         ENDIF
      IF (LUXLEV .LE. MAXLEV)  THEN
         NSKIP = NDSKIP(LUXLEV)
      ELSE
          NSKIP = LUXLEV - 24
          WRITE (6,'(A,I5)') ' RANLUX P-VALUE SET BY RLUXGO TO:',LUXLEV
      ENDIF
      IN24 = 0
      IF (INS .LT. 0) THEN  
        OPEN(UNIT=98,FILE='catfish-v2.10.warning',STATUS='UNKNOWN')
        WRITE(98,*) 'Warning: Illegal RSEED. Default value is used.'
        CLOSE(98)
      ENDIF	
      IF (INS .GT. 0)  THEN
        JSEED = INS
      ELSE
        JSEED = JSDFLT
        OPEN(UNIT=98,FILE='catfish-v2.10.warning',STATUS='UNKNOWN')
        WRITE(98,*) 'Warning: Default RSEED is used. '
        CLOSE(98)
      ENDIF
      INSEED = JSEED
      NOTYET = .FALSE.
      TWOM24 = 1.
         DO 325 I= 1, 24
           TWOM24 = TWOM24 * 0.5
         K = JSEED/53668
         JSEED = 40014*(JSEED-K*53668) -K*12211
         IF (JSEED .LT. 0)  JSEED = JSEED+ICONS
         ISEEDS(I) = MOD(JSEED,ITWO24)
  325    CONTINUE
      TWOM12 = TWOM24 * 4096.
         DO 350 I= 1,24
         SEEDS(I) = REAL(ISEEDS(I))*TWOM24
         NEXT(I) = I-1
  350    CONTINUE
      NEXT(1) = 24
      I24 = 24
      J24 = 10
      CARRY = 0.
      IF (SEEDS(24) .EQ. 0.) CARRY = TWOM24
      KOUNT = K1
      MKOUNT = K2
      IF (K1+K2 .NE. 0)  THEN
        DO 500 IOUTER= 1, K2+1
          INNER = IGIGA
          IF (IOUTER .EQ. K2+1)  INNER = K1
          DO 450 ISK= 1, INNER
            UNI = SEEDS(J24) - SEEDS(I24) - CARRY 
            IF (UNI .LT. 0.)  THEN
               UNI = UNI + 1.0
               CARRY = TWOM24
            ELSE
               CARRY = 0.
            ENDIF
            SEEDS(I24) = UNI
            I24 = NEXT(I24)
            J24 = NEXT(J24)
  450     CONTINUE
  500   CONTINUE
        IN24 = MOD(KOUNT, NSKIP+24)
        IF (MKOUNT .GT. 0)  THEN
           IZIP = MOD(IGIGA, NSKIP+24)
           IZIP2 = MKOUNT*IZIP + IN24
           IN24 = MOD(IZIP2, NSKIP+24)
        ENDIF
        IF (IN24 .GT. 23) THEN
           WRITE (6,'(A/A,3I11,A,I5)')  
     +    '  Error in RESTARTING with RLUXGO:','  The values', INS,
     +     K1, K2, ' cannot occur at luxury level', LUXLEV
           IN24 = 0
        ENDIF
      ENDIF
      RETURN
      END
C==========================================================
C     SUBROUTINE WRITESUMMARY
C==========================================================
      SUBROUTINE WRITESUMMARY()

      IMPLICIT NONE
      LOGICAL YNFLAG,EMCHARGEFLAG
      INTEGER NI,UP,NP,YVER,MOMTRANSF,RSEED,MLIST,NUMEVENTS
      DOUBLEPRECISION MST,ALPHA,PI,XMIN,QMIN,CUTS(6)
      DOUBLEPRECISION YYN(100),ZYN(100)
      
      CHARACTER*8 DATEI,DATEF
      CHARACTER*10 TIMEI,TIMEF
      CHARACTER*5 MONTH(12)
      CHARACTER*8 HOUR,DAY
      INTEGER TIMEARRAYI(8),TIMEARRAYF(8)
      REAL CPU, ETIME, ELAPSED(2)

      INTEGER KGENEV
      REAL TECM,AMASS(18)
 
      COMMON/GENIN / NP, TECM, AMASS, KGENEV

      
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/YOSHINO/ ZYN,YYN,UP,YVER
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
      COMMON/SIMULATION/CUTS,NUMEVENTS,RSEED,MLIST
      COMMON/DATETIME/DATEI,DATEF,TIMEARRAYI,TIMEARRAYF,CPU
      
      DATA MONTH/' Jan ',' Feb ',' Mar ',' Apr ',' May ',
     ! ' Jun ',' Jul ',' Aug ',' Sep ',' Oct ',' Nov ',
     ! ' Dec '/
 
      OPEN(UNIT=97,FILE='catfish-v2.10.sry',STATUS='UNKNOWN')


      WRITE(97,'(2(A))')
     !'\n *********************************************************',
     ''***\n'
      WRITE(97,'(2(A))')
     !'\n ******** ********* ********** ******** ** ******** **     **'
      WRITE(97,'(2(A))')
     !' ******** ********* ********** ******** ** ******** **     **'
      WRITE(97,'(2(A))')
     !' ** 	 **     **     **     **       ** **       **     ** '
      WRITE(97,'(2(A))')
     !' **	 *********     **     *****    ** *******  ********* '
      WRITE(97,'(2(A))')
     !' **	 *********     **     *****    **  ******* ********* '
      WRITE(97,'(2(A))')
     !' ** 	 **     **     **     **       **       ** **     ** '
      WRITE(97,'(2(A))')
     !' ******** **     **     **     **       ** ******** **     **'
      WRITE(97,'(2(A))')
     !' ******** **     **     **     **       ** ******** **     **\n'
      WRITE(97,'(2(A))') '       A Monte Carlo simulator for black',
     !' holes at the LHC'
      WRITE(97,'(A)')'                  The University of Mississippi'
      WRITE(97,'(2(A))')
     !'\n *********************************************************',
     ''***\n'
      WRITE(97,'(2(A))') '                  CATFISH ver. 2.10',
     ! ' - SUMMARY FILE\n'
      WRITE(97,'(A)')  ' PROCESSING PARAMETERS:\n'
       write(97,'(A,I2,A5,I4,3(A,I2),A)')
     ! ' Processing starting time:       ',TIMEARRAYI(3),
     ! MONTH(TIMEARRAYI(2)), TIMEARRAYI(1), '  ',
     ! TIMEARRAYI(5), ':',TIMEARRAYI(6),':',TIMEARRAYI(7)
      write(97,'(A,I2,A5,I4,3(A,I2),A)')
     ! ' Processing ending time:         ',TIMEARRAYF(3),
     ! MONTH(TIMEARRAYF(2)), TIMEARRAYF(1), '  ',
     ! TIMEARRAYF(5), ':',TIMEARRAYF(6),':',TIMEARRAYF(7)
      CPU = ETIME(ELAPSED)
      write(97,*)
     ! 'Total running time:           ', CPU, ' s'
      WRITE(97,*) 'Number of events:              ', NUMEVENTS
      write(97,*)
     ! 'Average cpu time per event:   ', CPU/NUMEVENTS, ' s'
      WRITE(97,*)    'Random seed:                   ', RSEED
      WRITE(97,*)    'Pythia PYLIST option:          ', MLIST
      WRITE(97,*)    'PT cuts (e,mu):               ', CUTS(1),
     ! 'GeV (e,mu) ', CUTS(2), ' GeV (gamma,jet)'
     
      WRITE(97,*)'Dilepton PT cut:              ',CUTS(3),' GeV'
      WRITE(97,*)'Dilepton pseudorapidity cut:  ',CUTS(4)
      WRITE(97,*)'Dilepton isolation cut, PT <=:',CUTS(5),' GeV'
      WRITE(97,*)'Within a cone of R:           ',CUTS(6)
      
      WRITE(97,'(A)')  '\n BLACK HOLE PARAMETERS:\n'
      WRITE(97,*)    'Fundamental Scale:            ', MST, ' TeV'
      WRITE(97,*)    'Number of extra dimensions:    ', NI
      IF (YNFLAG .EQV. .TRUE.) THEN
       WRITE(97,'(A)') 
     ! ' Black hole formation model:     TRAPPED SURFACE'
      ELSE
       WRITE(97,'(A)')
     ! ' Black hole formation model:     BLACK DISK'
      ENDIF
      IF (YNFLAG .EQV. .TRUE.) THEN
        IF (YVER .EQ. 0) THEN
	 WRITE(97,'(A)') ' Trapped-surface model:          YOSHINO-NAMBU'
	ELSE
	 WRITE(97,'(A)') ' Trapped-surface model:          YOSHINO-RYCHKOV'
	ENDIF
      ENDIF	
      WRITE(97,*) 'Minimum BH mass:              ',
     ! XMIN*MST,' TeV'
      WRITE(97,*) 'Quantum BH mass threshold:    ',
     ! QMIN*MST,' TeV'
      IF (NP .EQ. 0) THEN
       WRITE(97,*) 'Final decay product:            ',
     ! 'BH REMNANT'
      ELSE
       WRITE(97,*) 'Final decay product:           ',
     ! NP, ' HARD QUANTA'
      ENDIF
      IF (MOMTRANSF .EQ. 1) THEN
       WRITE(97,'(2(A))')' Momentum transfer:              ',
     ! 'BH MASS'
      ELSE
       WRITE(97,'(2(A))')' Momentum transfer:              ',
     ! 'SCHW. RADIUS INVERSE'
      ENDIF      
      IF (EMCHARGEFLAG .EQV. .TRUE.) THEN
       WRITE(97,'(A)') ' Conservation of EM charge:      YES'
      ELSE 
       WRITE(97,'(A)') ' Conservation of EM charge:      NO'
      ENDIF
      IF (ALPHA .EQ. 0.0D0) THEN
       WRITE(97,*)   'Minimum length:                 ',
     ! 'NO'
      ELSE
       WRITE(97,*)   'Minimum length:               ',
     ! ALPHA*2.0D0*MST, ' TeV^{-1}'
      ENDIF
      WRITE(97,'(2(A))')
     !'\n *********************************************************',
     ''***\n'
      CLOSE(97)

      RETURN
      END
C==================================================================== 
c     Subroutine UPINIT
C     Purpose: Initializes the event record
C==================================================================== 
      SUBROUTINE UPINIT
 
      IMPLICIT DOUBLE PRECISION(A-H, O-Z)
      IMPLICIT INTEGER(I-N)
 
      COMMON/PYDAT1/MSTU(200),PARU(200),MSTJ(200),PARJ(200)
      COMMON/PYPARS/MSTP(200),PARP(200),MSTI(200),PARI(200)
      SAVE /PYDAT1/,/PYPARS/
 
      INTEGER MAXPUP
      PARAMETER (MAXPUP=100)
      INTEGER IDBMUP,PDFGUP,PDFSUP,IDWTUP,NPRUP,LPRUP
      DOUBLE PRECISION EBMUP,XSECUP,XERRUP,XMAXUP
      COMMON/HEPRUP/IDBMUP(2),EBMUP(2),PDFGUP(2),PDFSUP(2),
     &IDWTUP,NPRUP,XSECUP(MAXPUP),XERRUP(MAXPUP),XMAXUP(MAXPUP),
     &LPRUP(MAXPUP)
      SAVE /HEPRUP/
 
      LOGICAL YNFLAG,EMCHARGEFLAG
      
      DOUBLEPRECISION LHCCROSS,ECM
      EXTERNAL LHCCROSS
 
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
      COMMON/IENERGY/ECM

      CHARACTER CHAF*16
      COMMON/PYDAT4/CHAF(500,2)
      SAVE/PYDAT4/

      CHAF(80,1)='BH remnant'
      CHAF(79,1)='Initial GRW'
      
      IDBMUP(1)=2212
      IDBMUP(2)=2212
      
      EBMUP(1)=ECM*500.0D0
      EBMUP(2)=ECM*500.0D0    

      PDFGUP(1)=4
      PDFGUP(2)=4
      
      PDFSUP(1)=53
      PDFSUP(2)=53
      
      MSTP(51)=8
                  
      IDWTUP=3
      
      NPRUP=1 

      XSECUP(1)=LHCCROSS()*19.733D0**2.0D0

      LPRUP(1) = 999

      RETURN
      END
C==================================================================== 
C     Subroutine UPEVNT
C     Purpose: Generates the event record
C====================================================================
      SUBROUTINE UPEVNT
 
      INTEGER MSTU(200),MSTJ(200),MSTP(200),MSTI(200)
      DOUBLEPRECISION PARU(200),PARJ(200),PARP(200),PARI(200)
 
      COMMON/PYDAT1/MSTU,PARU,MSTJ,PARJ
      COMMON/PYPARS/MSTP,PARP,MSTI,PARI
      SAVE /PYDAT1/,/PYPARS/
 
      INTEGER MAXNUP
      PARAMETER (MAXNUP=500)
      INTEGER NUP,IDPRUP,IDUP,ISTUP,MOTHUP,ICOLUP
      DOUBLE PRECISION XWGTUP,SCALUP,AQEDUP,AQCDUP,PUP,VTIMUP,SPINUP
      COMMON/HEPEUP/NUP,IDPRUP,XWGTUP,SCALUP,AQEDUP,AQCDUP,IDUP(MAXNUP),
     &ISTUP(MAXNUP),MOTHUP(2,MAXNUP),ICOLUP(2,MAXNUP),PUP(5,MAXNUP),
     &VTIMUP(MAXNUP),SPINUP(MAXNUP)
      SAVE /HEPEUP/


      INTEGER HARDPARTICLE(18,3),TABLEPARTICLE(100,3),BHARRAY(1000),IBH
      DOUBLE PRECISION S,B,TABLEMASS(100)
      DOUBLE PRECISION REALBMAX,VP(1000,6)
      REAL RVEC,RVEC2
      
      INTEGER NI,MOMTRANSF
      DOUBLE PRECISION MST,ALPHA,XMIN,QMIN,ECM
      COMMON/BH_PARAMETERS/MST,ALPHA,XMIN,QMIN,NI,MOMTRANSF
      COMMON/IENERGY/ECM
      COMMON/BHMASS/M
      
      DOUBLE PRECISION MBH,T,M,RSCHWA,D,LAMBDA

      EXTERNAL LAMBDA 

      REAL TECM,AMASS(18),PCM(5,18),WT
      INTEGER I,J,J2,NUMPARTICLE,KGENEV,NP,EMCHARGE,JJ
      INTEGER NTEMP


      COMMON/GENIN/ NP, TECM, AMASS, KGENEV
      COMMON/GENOUT/ PCM, WT
      
      INTEGER FLAVOR1,FLAVOR2,ICLEFT,ICRIGHT,ICOLMTX(2,100)

      INTEGER MAXPUP
      PARAMETER (MAXPUP=100)
      INTEGER IDBMUP,PDFGUP,PDFSUP,IDWTUP,NPRUP,LPRUP
      DOUBLE PRECISION EBMUP,XSECUP,XERRUP,XMAXUP
      COMMON/HEPRUP/IDBMUP(2),EBMUP(2),PDFGUP(2),PDFSUP(2),
     &IDWTUP,NPRUP,XSECUP(MAXPUP),XERRUP(MAXPUP),XMAXUP(MAXPUP),
     &LPRUP(MAXPUP)
     
      DOUBLEPRECISION PYMASS
      EXTERNAL PYMASS

      DOUBLEPRECISION PUPC(4),PUPL(4)
      DOUBLEPRECISION BOOST
      DOUBLEPRECISION UX,UY,UZ

      DOUBLEPRECISION QTHRESHOLD
      EXTERNAL QTHRESHOLD

      LOGICAL YNFLAG,EMCHARGEFLAG
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG

      IDPRUP=999      

      ISTUP(1)=-1
      ISTUP(2)=-1
      
        
50    ICLEFT=0
      ICRIGHT=0
       
      DO I=1,2
        DO J=1,100
	ICOLMTX(I,J)=0
	ENDDO
      ENDDO	
                             
      CALL RANLUX(RVEC, 4)
 
      CALL MASSREDUCE(S)    

	
      B=DBLE(RVEC)*REALBMAX(S)

      D=DBLE(NI+4)
      M=MBH(B,S)
      RSCHWA = LAMBDA()*(M/MST)**(1.0D0/(D-3.0D0))/MST	  	
	
	      
      IF (MOMTRANSF .EQ. 1) THEN		

	  IF (M .LT. 10.0D0) THEN
	      SCALUP = M * 1000.D0
	  ELSE
	      SCALUP = 10000.D0
	  ENDIF
	  	
      ELSEIF (MOMTRANSF .EQ. 2) THEN  	  

	  IF (1.D0/RSCHWA .LT. 10.0D0) THEN
	      SCALUP = 1.0D0/RSCHWA * 1000.D0
	  ELSE
	      SCALUP = 10000.D0
	  ENDIF

      ENDIF
      

100   CALL VECTORPARTICLE(S,B,TABLEPARTICLE,HARDPARTICLE,
     !                      NUMPARTICLE,VP,EMCHARGE)

      NTEMP=NUMPARTICLE+NP
      NUP=NTEMP+2

 	 CALL RANLUX(RVEC,4)
         FLAVOR1=INT(3.*RVEC)
	 CALL RANLUX(RVEC,4)
         FLAVOR2=INT(3.*RVEC)	 


      IF (EMCHARGE .EQ. -4) THEN
	 IDUP(1)=-2-2*FLAVOR1
	 IDUP(2)=-2-2*FLAVOR2
	 ICOLUP(1,1)=0
         ICOLUP(2,1)=1
	 ICOLMTX(2,1)=ICOLMTX(2,1)+1
         ICOLUP(1,2)=0
         ICOLUP(2,2)=2      
	 ICOLMTX(2,2)=ICOLMTX(2,2)+1
      ELSEIF (EMCHARGE .EQ. -3) THEN      
	 IDUP(1)=-2-2*FLAVOR1
	 IDUP(2)=1+2*FLAVOR2
	 ICOLUP(1,1)=0
         ICOLUP(2,1)=1
	 ICOLMTX(2,1)=ICOLMTX(2,1)+1
	 CALL RANLUX(RVEC,4)
         IF (2.*RVEC .LT. 1.) THEN
	     ICOLUP(1,2)=1
             ICOLUP(2,2)=0
	     ICOLMTX(1,1)=ICOLMTX(1,1)+1
	 ELSE
	     ICOLUP(1,2)=2
             ICOLUP(2,2)=0    
	     ICOLMTX(1,2)=ICOLMTX(1,2)+1
	 ENDIF
      ELSEIF (EMCHARGE .EQ. -2) THEN      
         CALL RANLUX(RVEC,4)
         IF (2.*RVEC .LT. 1.) THEN           
	   IDUP(1)=-2-2*FLAVOR1
	   IDUP(2)=21	  
           ICOLUP(1,1)=0
           ICOLUP(2,1)=1
	   ICOLMTX(2,1)=ICOLMTX(2,1)+1
	   CALL RANLUX(RVEC2,4)
	   IF (2.*RVEC2 .LT. 1.) THEN
	     ICOLUP(1,2)=1
             ICOLUP(2,2)=2
	     ICOLMTX(1,1)=ICOLMTX(1,1)+1
	     ICOLMTX(2,2)=ICOLMTX(2,2)+1
	   ELSE
	     ICOLUP(1,2)=2
             ICOLUP(2,2)=3    
	     ICOLMTX(1,2)=ICOLMTX(1,2)+1
	     ICOLMTX(2,3)=ICOLMTX(2,3)+1
	   ENDIF
	 ELSE
	   IDUP(1)=1+2*FLAVOR1
	   IDUP(2)=1+2*FLAVOR2
           ICOLUP(1,1)=1
           ICOLUP(2,1)=0
           ICOLUP(1,2)=2
           ICOLUP(2,2)=0
	   ICOLMTX(1,1)=ICOLMTX(1,1)+1
	   ICOLMTX(1,2)=ICOLMTX(1,2)+1
	 ENDIF
      ELSEIF (EMCHARGE .EQ. -1) THEN
      CALL RANLUX(RVEC,4)
        IF (2.*RVEC .LT. 1) THEN           
 	  IDUP(1)=1+2*FLAVOR1
	  IDUP(2)=21
          ICOLUP(1,1)=1
          ICOLUP(2,1)=0
	  ICOLMTX(1,1)=ICOLMTX(1,1)+1
	  CALL RANLUX(RVEC2,4)
	  IF (2.*RVEC2 .LT. 1.) THEN
            ICOLUP(1,2)=2
            ICOLUP(2,2)=3
	    ICOLMTX(1,2)=ICOLMTX(1,2)+1
	    ICOLMTX(2,3)=ICOLMTX(2,3)+1	  
	  ELSE
	    ICOLUP(1,2)=2
            ICOLUP(2,2)=1
	    ICOLMTX(1,2)=ICOLMTX(1,2)+1
	    ICOLMTX(2,1)=ICOLMTX(2,1)+1
	  ENDIF        
	ELSE
 	  IDUP(1)=-2-2*FLAVOR1
 	  IDUP(2)=-1-2*FLAVOR2
          ICOLUP(1,1)=0
          ICOLUP(2,1)=1
          ICOLUP(1,2)=0
          ICOLUP(2,2)=2
	  ICOLMTX(2,1)=ICOLMTX(2,1)+1
	  ICOLMTX(2,2)=ICOLMTX(2,2)+1
        ENDIF
      ELSEIF (EMCHARGE .EQ. 0) THEN
      CALL RANLUX(RVEC,4)
        IF (2.*RVEC .LT. 1.) THEN           
	   IDUP(1)=21
 	   IDUP(2)=21
           ICOLUP(1,1)=1
           ICOLUP(2,1)=2
	   ICOLMTX(1,1)=ICOLMTX(1,1)+1
	   ICOLMTX(2,2)=ICOLMTX(2,2)+1
           CALL RANLUX(RVEC2,4)
	   IF (3.*RVEC2 .LT. 1.) THEN
            ICOLUP(1,2)=2
            ICOLUP(2,2)=1
	    ICOLMTX(1,2)=ICOLMTX(1,2)+1
	    ICOLMTX(2,1)=ICOLMTX(2,1)+1	  
          ELSEIF (3.*RVEC2 .LT. 2.) THEN
	    ICOLUP(1,2)=2
            ICOLUP(2,2)=3
	    ICOLMTX(1,2)=ICOLMTX(1,2)+1
	    ICOLMTX(2,3)=ICOLMTX(2,3)+1
	  ELSE
	    ICOLUP(1,2)=3
            ICOLUP(2,2)=1
	    ICOLMTX(1,3)=ICOLMTX(1,3)+1
	    ICOLMTX(2,1)=ICOLMTX(2,1)+1
	  ENDIF        
       ELSE
          ICOLUP(1,1)=0
          ICOLUP(2,1)=1
	  ICOLMTX(2,1)=ICOLMTX(2,1)+1
          CALL RANLUX(RVEC2,4)
	  IF (2.*RVEC2 .LT. 1) THEN
	    IDUP(1)=-2-2*FLAVOR1
	    IDUP(2)=2+2*FLAVOR2
            ICOLUP(1,2)=1
            ICOLUP(2,2)=0
	    ICOLMTX(1,1)=ICOLMTX(1,1)+1	     
	  ELSE
	    IDUP(1)=-1-2*FLAVOR1
	    IDUP(2)=1+2*FLAVOR2
            ICOLUP(1,2)=2
            ICOLUP(2,2)=0
	    ICOLMTX(1,2)=ICOLMTX(1,2)+1	     
	  ENDIF
	ENDIF  
      ELSEIF (EMCHARGE .EQ. 1) THEN
      CALL RANLUX(RVEC,4)
        IF (2.*RVEC .LT. 1) THEN                 
 	 IDUP(1)=-1-2*FLAVOR1
 	 IDUP(2)=21
         ICOLUP(1,1)=0
         ICOLUP(2,1)=1
	 ICOLMTX(2,1)=ICOLMTX(2,1)+1
	 CALL RANLUX(RVEC2,4)
         IF (2.*RVEC2 .LT. 1) THEN
           ICOLUP(1,2)=1
           ICOLUP(2,2)=2
	   ICOLMTX(1,1)=ICOLMTX(1,1)+1
 	   ICOLMTX(2,2)=ICOLMTX(2,2)+1
         ELSE
           ICOLUP(1,2)=2
           ICOLUP(2,2)=3
	   ICOLMTX(1,2)=ICOLMTX(1,2)+1
 	   ICOLMTX(2,3)=ICOLMTX(2,3)+1
         ENDIF
        ELSE
	 IDUP(1)=2+2*FLAVOR1
	 IDUP(2)=1+2*FLAVOR2
         ICOLUP(1,1)=1
         ICOLUP(2,1)=0 
	 ICOLMTX(1,1)=ICOLMTX(1,1)+1
         ICOLUP(1,2)=2
         ICOLUP(2,2)=0 
	 ICOLMTX(1,2)=ICOLMTX(1,2)+1
	ENDIF
      ELSEIF (EMCHARGE .EQ. 2) THEN      
         CALL RANLUX(RVEC,4)
         IF (2.*RVEC .LT. 1.) THEN           
	   IDUP(1)=2+2*FLAVOR1
	   IDUP(2)=21	  
           ICOLUP(1,1)=1
           ICOLUP(2,1)=0
	   ICOLMTX(1,1)=ICOLMTX(1,1)+1
	   CALL RANLUX(RVEC2,4)
	   IF (2.*RVEC2 .LT. 1.) THEN
	     ICOLUP(1,2)=2
             ICOLUP(2,2)=1
	     ICOLMTX(1,2)=ICOLMTX(1,2)+1
	     ICOLMTX(2,1)=ICOLMTX(2,1)+1
	   ELSE
	     ICOLUP(1,2)=2
             ICOLUP(2,2)=3    
	     ICOLMTX(1,2)=ICOLMTX(1,2)+1
	     ICOLMTX(2,3)=ICOLMTX(2,3)+1
	   ENDIF
	 ELSE
	   IDUP(1)=-1-2*FLAVOR1
	   IDUP(2)=-1-2*FLAVOR2
           ICOLUP(1,1)=0
           ICOLUP(2,1)=1
           ICOLUP(1,2)=0
           ICOLUP(2,2)=2
	   ICOLMTX(2,1)=ICOLMTX(2,1)+1
	   ICOLMTX(2,2)=ICOLMTX(2,2)+1
	 ENDIF
      ELSEIF (EMCHARGE .EQ. 3) THEN      
	 IDUP(1)=2+2*FLAVOR1
	 IDUP(2)=-1-2*FLAVOR2
	 ICOLUP(1,1)=1
         ICOLUP(2,1)=0
	 ICOLMTX(1,1)=ICOLMTX(1,1)+1
	 CALL RANLUX(RVEC,4)
         IF (2.*RVEC .LT. 1.) THEN
	     ICOLUP(1,2)=0
             ICOLUP(2,2)=1
	     ICOLMTX(2,1)=ICOLMTX(2,1)+1
	 ELSE
	     ICOLUP(1,2)=0
             ICOLUP(2,2)=2    
	     ICOLMTX(2,2)=ICOLMTX(2,2)+1
	 ENDIF
      ELSEIF (EMCHARGE .EQ. 4) THEN
	 IDUP(1)=2+2*FLAVOR1
	 IDUP(2)=2+2*FLAVOR2
	 ICOLUP(1,1)=1
         ICOLUP(2,1)=0
	 ICOLMTX(1,1)=ICOLMTX(1,1)+1
         ICOLUP(1,2)=2
         ICOLUP(2,2)=0      
	 ICOLMTX(1,2)=ICOLMTX(1,2)+1
      ELSE
         STOP      
      ENDIF
      

      IF (ICOLUP(1,1) .NE. 0 .AND. ICOLUP(1,1) .EQ. ICOLUP(2,2))
     ! THEN
	  ICOLMTX(1,ICOLUP(1,1))=0
 	  ICOLMTX(2,ICOLUP(2,2))=0	  
      ENDIF
      
      IF (ICOLUP(1,2) .NE. 0 .AND. ICOLUP(1,2) .EQ. ICOLUP(2,1))
     ! THEN
 	  ICOLMTX(1,ICOLUP(1,2))=0
 	  ICOLMTX(2,ICOLUP(2,1))=0
      ENDIF
       

      DO J=1,2
        IF (IDUP(J) .EQ. 6) THEN
	   CALL RANLUX(RVEC,4)
	   IDUP(J)=2+2*INT(2.*RVEC)
	ELSEIF (IDUP(J) .EQ. -6) THEN
	   CALL RANLUX(RVEC,4)
	   IDUP(J)=-2-2*INT(2.*RVEC)
	ENDIF
      ENDDO	   


      PUP(1,1)=0.
      PUP(2,1)=0.
      PUP(1,2)=0.
      PUP(2,2)=0.
      PUP(5,1)=0.
      PUP(5,2)=0.

      PUP(3,1)=DSQRT(S)*500.0D0
      PUP(4,1)=PUP(3,1)
      
      PUP(3,2)=-DSQRT(S)*500.0D0
      PUP(4,2)=-PUP(3,2)


      DO I=1, ntemp
      ISTUP(I+2)=1      
      IDUP(I+2)=IDINT(VP(I,1))
      MOTHUP(1,I+2)=1
      MOTHUP(2,I+2)=2                  
      IF (IDUP(I+2) .GT. 0 .AND. IDUP(I+2) .LE. 6) THEN 
	DO J=1,3
	  IF (ICOLMTX(1,J) .GT. 0) THEN
	  ICOLUP(1,I+2)=J
	  ICOLUP(2,I+2)=0
	  ICOLMTX(1,J)=ICOLMTX(1,J)-1
	  GOTO 400
	  ENDIF
	ENDDO
	DO J=4,100
	  IF (ICOLMTX(2,J) .GT. 0) THEN
	  ICOLUP(1,I+2)=J
	  ICOLUP(2,I+2)=0
	  ICOLMTX(2,J)=ICOLMTX(2,J)-1
	  GOTO 400
	  ENDIF
	ENDDO
	  ICOLUP(1,I+2)=3+I
	  ICOLUP(2,I+2)=0
	  ICOLMTX(1,3+I)=ICOLMTX(1,3+I)+1
	GOTO 400
      ELSEIF (IDUP(I+2) .GE. -6 .AND. IDUP(I+2) .LT. 0) THEN
	DO J=1,3
	  IF (ICOLMTX(2,J) .GT. 0) THEN
	  ICOLUP(1,I+2)=0
	  ICOLUP(2,I+2)=J
	  ICOLMTX(2,J)=ICOLMTX(2,J)-1
	  GOTO 400
	  ENDIF
	ENDDO
	DO J=4,100
	  IF (ICOLMTX(1,J) .GT. 0) THEN
	  ICOLUP(1,I+2)=0
	  ICOLUP(2,I+2)=J
	  ICOLMTX(1,J)=ICOLMTX(1,J)-1
	  GOTO 400
	  ENDIF
	ENDDO
	  ICOLUP(1,I+2)=0
	  ICOLUP(2,I+2)=3+I
	  ICOLMTX(2,3+I)=ICOLMTX(2,3+I)+1
	GOTO 400
      ELSEIF (IDUP(I+2) .EQ. 21) THEN


	DO J=1,3
	  IF (ICOLMTX(1,J) .GT. 0) THEN
	  ICOLUP(1,I+2)=J
	  ICOLMTX(1,J)=ICOLMTX(1,J)-1
	   DO J2=1,3
	   IF (ICOLMTX(2,J2) .GT. 0) THEN
	      ICOLUP(2,I+2)=J2
	      ICOLMTX(2,J2)=ICOLMTX(2,J2)-1
	      GOTO 400
	   ENDIF   
	   ENDDO
	   DO J2=4,100
	   IF (ICOLMTX(1,J2) .GT. 0) THEN
	      ICOLUP(2,I+2)=J2
	      ICOLMTX(1,J2)=ICOLMTX(1,J2)-1
	      GOTO 400
	   ENDIF
	   ENDDO
	   ICOLUP(2,I+2)=I+3
	   ICOLMTX(2,I+3)=ICOLMTX(2,I+3)+1
          GOTO 400
	  ENDIF
	ENDDO


	DO J=1,3
	  IF (ICOLMTX(2,J) .GT. 0) THEN
	  ICOLUP(2,I+2)=J
	  ICOLMTX(2,J)=ICOLMTX(2,J)-1
	   DO J2=4,100
	   IF (ICOLMTX(2,J2) .GT. 0) THEN
	      ICOLUP(1,I+2)=J2
	      ICOLMTX(2,J2)=ICOLMTX(2,J2)-1
	      GOTO 400
	   ENDIF
	   ENDDO
	   ICOLUP(1,I+2)=I+3
	   ICOLMTX(1,I+3)=ICOLMTX(1,I+3)+1
          GOTO 400
	  ENDIF
	ENDDO


 	DO J=4,100
	  IF (ICOLMTX(1,J) .GT. 0) THEN
	  ICOLUP(2,I+2)=J
	  ICOLMTX(1,J)=ICOLMTX(1,J)-1
	  
	  do jj=4,100
	  IF (ICOLMTX(2,Jj) .GT. 0) THEN
	  ICOLUP(1,I+2)=jj
	  ICOLMTX(2,jj)=ICOLMTX(2,jj)-1
	  goto 400	  
	  endif
	  enddo
	  
	  ICOLUP(1,I+2)=I+3
	  ICOLMTX(1,I+3)=ICOLMTX(1,I+3)+1

	  GOTO 400
	  ENDIF
	  IF (ICOLMTX(2,J) .GT. 0) THEN
	  ICOLUP(1,I+2)=J
	  ICOLMTX(2,J)=ICOLMTX(2,J)-1
	  ICOLUP(2,I+2)=I+3
	  ICOLMTX(2,I+3)=ICOLMTX(2,I+3)+1
	  GOTO 400
	  ENDIF
	ENDDO  
	   ICOLUP(1,I+2)=I+3
	   ICOLMTX(1,I+3)=ICOLMTX(1,I+3)+1
	   ICOLUP(2,I+2)=I+4
	   ICOLMTX(2,I+4)=ICOLMTX(1,I+4)+1
          GOTO 400
        ELSE
	  ICOLUP(1,I+2)=0
	  ICOLUP(2,I+2)=0
          GOTO 400
	ENDIF    
	
400     CONTINUE

      PUP(1,I+2)=VP(I,3)*1000.
      PUP(2,I+2)=VP(I,4)*1000.
      PUP(3,I+2)=VP(I,5)*1000.
      PUP(4,I+2)=VP(I,2)*1000.
      PUP(5,I+2)=PYMASS(IDUP(I+2))

      ENDDO

            
      IF (YNFLAG .EQV. .TRUE.) THEN
        DO J=1,2
        NUP=NUP+1
	ISTUP(NUP)=1
	IDUP(NUP)=79
        PUP(1,NUP)=0.0D0
        PUP(2,NUP)=0.0D0
        PUP(4,NUP)=(DSQRT(S)-M)*500.0D0
        PUP(5,NUP)=0.0D0
	ICOLUP(1,NUP)=0
	ICOLUP(2,NUP)=0
	ENDDO
      PUP(3,NUP-1)=(DSQRT(S)-M)*500.0D0
      PUP(3,NUP)=-(DSQRT(S)-M)*500.0D0
      ENDIF
      
      IF (NP .EQ. 0) THEN      
	ISTUP(NUP+1)=1
	IDUP(NUP+1)=IDINT(VP(I,1))
        PUP(1,NUP+1)=VP(NTEMP+1,3)*1000.
        PUP(2,NUP+1)=VP(NTEMP+1,4)*1000.
        PUP(3,NUP+1)=VP(NTEMP+1,5)*1000.
        PUP(4,NUP+1)=VP(NTEMP+1,2)*1000.
        IF (NUMPARTICLE .EQ. 0) THEN
	   PUP(5,NUP+1)=PUP(4,NUP+1)
	ELSE
	   PUP(5,NUP+1)=QTHRESHOLD()*1000.
	ENDIF   	
	ICOLUP(1,NUP+1)=0
	ICOLUP(2,NUP+1)=0
1200    NUP=NUP+1
      ENDIF

      
      DO J=1,100
	ICLEFT=ICLEFT+ICOLMTX(1,J)
	ICRIGHT=ICRIGHT+ICOLMTX(2,J)
      ENDDO	
	
      IF(ICLEFT .NE. 0 .OR. ICRIGHT .NE. 0) GOTO 50
                              
      RETURN
      END

C====================================================================
C     SUBROUTINE EVTEN
C     Purpose: Gives the various energy/PT of the event:
C====================================================================
      SUBROUTINE EVTEN(ENEVT,PTEVT)

      INTEGER N,NPAD,K(4000,5),KK,PYK,I,NUMEVENTS,RSEED,MLIST
      INTEGER EP,EN,MUP,MUN,TAUP,TAUN,AIE,AIM
      INTEGER CTR_MU,CTR_LEP,CNT,T,TOTLEP
      DOUBLEPRECISION P(4000,5),V(4000,5),PYP,ENEVT(11),CUTS(6)
      DOUBLEPRECISION ENEVTM,ECM,PTEVT(11),SUMET1,
     !PXEVT,PYEVT,DOTPR,MLL
     
      COMMON/PYJETS/N,NPAD,K,P,V
      COMMON/IENERGY/ECM
      
      LOGICAL YNFLAG,EMCHARGEFLAG
      COMMON/FLAGS/YNFLAG,EMCHARGEFLAG
      COMMON/SIMULATION/CUTS,NUMEVENTS,RSEED,MLIST

      EXTERNAL PYK,PYP


      CTR_MU=0
      CTR_LEP=0
      PXEVT=0.0D0
      PYEVT=0.0D0	
      EP=0
      EN=0
      MUP=0
      MUN=0
      TAUP=0
      TAUN=0
      AIE=0
      AIM=0

	DO I=1,11
	   ENEVT(I)=0.0D0
	   PTEVT(I)=0.0D0
	ENDDO
	
	KK=PYK(0,1)


	DO I=1,KK

	IF (PYK(I,2) .EQ. 80) GOTO 200

	IF (PYK(I,2) .EQ. 79) THEN
	    ENEVT(3)=ENEVT(3)+PYP(I,4)
	    GOTO 200
	ENDIF


	IF (PYK(I,2) .EQ. 39) THEN
	    ENEVT(5)=ENEVT(5)+PYP(I,4)
	    PTEVT(3)=PTEVT(3)+PYP(I,10)
	    GOTO 200
	ENDIF
	

	IF(IABS(PYK(I,2)) .EQ. 12
     !	   .OR. IABS(PYK(I,2)) .EQ. 14
     !     .OR. IABS(PYK(I,2)) .EQ. 16) THEN
	    ENEVT(6)=ENEVT(6)+PYP(I,4)
	    PTEVT(4)=PTEVT(4)+PYP(I,10)
            GOTO 200
	ENDIF    

        IF(IABS(PYK(I,2)) .EQ. 11 .OR. IABS(PYK(I,2)) .EQ. 13)
     !	THEN
           IF (PYP(I,10) .LT. CUTS(1)) THEN
	       ENEVT(8)=ENEVT(8)+PYP(I,4)
	   ELSE
	       ENEVT(2)=ENEVT(2)+PYP(I,4)
	       ENEVT(10)=ENEVT(10)+PYP(I,4)
	       PTEVT(2)=PTEVT(2)+PYP(I,10)	
	       PTEVT(6)=PTEVT(6)+PYP(I,10)	         
	       PXEVT=PXEVT+PYP(I,1)
	       PYEVT=PYEVT+PYP(I,2)
	   ENDIF
	   GOTO 200    
	ENDIF 

        IF(PYP(I,10) .LT. CUTS(2)) THEN
	    ENEVT(9)=ENEVT(9)+PYP(I,4)
	    ELSE
	       ENEVT(2)=ENEVT(2)+PYP(I,4)
	       ENEVT(11)=ENEVT(11)+PYP(I,4)
	       PTEVT(2)=PTEVT(2)+PYP(I,10)	
	       PTEVT(7)=PTEVT(7)+PYP(I,10)	         
	       PXEVT=PXEVT+PYP(I,1)
	       PYEVT=PYEVT+PYP(I,2)
	ENDIF 

200     CONTINUE
	ENDDO

	ENEVT(1)=ECM*1000.0D0-ENEVT(2)
	ENEVT(4)=ENEVT(5)+ENEVT(6)
	ENEVT(7)=ENEVT(8)+ENEVT(9)

	PTEVT(1)=DSQRT(PXEVT*PXEVT+PYEVT*PYEVT)
	PTEVT(5)=PTEVT(3)+PTEVT(4)

	RETURN
	END
C====================================================================
C     SUBROUTINE OSSFDILEPMLL
C     Purpose: Calculates the opposite sign, same flavor (OSSF)
C              dilepton invariant mass.
C====================================================================
      SUBROUTINE OSSFDILEPMLL(IARR,DMLL)
      DOUBLEPRECISION PYP,P(4000,5),V(4000,5),INVMAS,MLL,DOTPR,MLCHI10,
     !ELEC(20,6),MUON(20,6),DELTA_R,DMLL(4),SUMET1,SUMET2,PTCUT,ETACUT
     !SUMETCUT,RCUT,PTE,PTMU,CUTS(6)
      INTEGER IARR(5),PYK,N,NPAD,K(4000,5),I,J,FLAG,CFLAG,SFLAG,EN,EP,
     !MUN,MUP,AIE,AIM,T,CNT,TOTLEP,JETCTR,KK,KK1,NUMEVENTS,RSEED,MLIST
      COMMON/PYJETS/N,NPAD,K,P,V
      COMMON/SIMULATION/CUTS,NUMEVENTS,RSEED,MLIST   
      EXTERNAL PYK,PYP
      
      PTCUT=CUTS(3)
      ETACUT=CUTS(4)
      SUMETCUT=CUTS(5)
      RCUT=CUTS(6)
      
      TOTLEP=0
      AIE=0
      AIM=0
      JETCTR=0
      SUMET1=0.0D0
      SUMET2=0.0D0
      IARR(1)=0
      IARR(2)=0
      IARR(3)=0
      IARR(4)=0
      IARR(5)=0
      
      FLAG=0
      CFLAG=0
      SFLAG=0
      DMLL(1)=0.0D0
      DMLL(2)=0.0D0
      DMLL(3)=0.0D0
      DMLL(4)=0.0D0
      PTE=0.0D0
      PTMU=0.0D0
      
      MLL=0.0D0
      DELTA_R=0.0D0

	EP=0
	EN=0
	MUP=0
	MUN=0
      
      DO T=1,20
	  DO CNT=1,6
		ELEC(T,CNT)=0.0D0
		MUON(T,CNT)=0.0D0
	  ENDDO
      ENDDO       
      
      KK=PYK(0,1)
      KK1=KK
      
      DO I=1,KK                 
      IF(IABS(PYK(I,2)) .EQ. 11 .AND. PYP(I,10) .GE. PTCUT
     !.AND. DABS(PYP(I,19)) .LE. ETACUT) THEN
          DO T=1,KK1
	    DELTA_R=DSQRT((PYP(I,19)-PYP(T,19))**2.0D0+(PYP(I,15)-PYP(T,15))**2.0D0)
	    IF(DELTA_R .GT. 0.0D0 .AND. DELTA_R .LE. RCUT) THEN
	      SUMET1=SUMET1+PYP(T,10)	
	    ENDIF
	  ENDDO
	  IF(SUMET1 .LT. SUMETCUT) THEN
      	    IF((PYK(I,2)) .EQ. -11) EN=EN+1
      	    IF((PYK(I,2)) .EQ. 11) EP=EP+1
      	    AIE=AIE+1
      	    ELEC(AIE,1)=PYP(I,1)
     	    ELEC(AIE,2)=PYP(I,2)
      	    ELEC(AIE,3)=PYP(I,3)
      	    ELEC(AIE,4)=PYP(I,4)
            ELEC(AIE,5)=PYP(I,19)
	    ELEC(AIE,6)=PYP(I,15) 
	    PTE=PTE+PYP(I,10)
         ENDIF
       ENDIF
       
      IF(IABS(PYK(I,2)) .EQ. 13 .AND. PYP(I,10) .GE. PTCUT
     !.AND.  DABS(PYP(I,19)) .LE. ETACUT) THEN 
          DO T=1,KK1
	    DELTA_R=DSQRT((PYP(I,19)-PYP(T,19))**2.0D0+(PYP(I,15)-PYP(T,15))**2.0D0)
	    IF(DELTA_R .GT. 0.0D0 .AND. DELTA_R .LE. RCUT) THEN
	      SUMET2=SUMET2+PYP(T,10)
	    ENDIF
	  ENDDO
	  IF(SUMET2 .LT. SUMETCUT) THEN  
      	    IF((PYK(I,2)) .EQ. -13) MUN=MUN+1
      	    IF((PYK(I,2)) .EQ. 13) MUP=MUP+1
      	    AIM=AIM+1
      	    MUON(AIM,1)=PYP(I,1)
      	    MUON(AIM,2)=PYP(I,2)
      	    MUON(AIM,3)=PYP(I,3)
      	    MUON(AIM,4)=PYP(I,4)
            MUON(AIM,5)=PYP(I,19)
	    MUON(AIM,6)=PYP(I,15)
	    PTMU=PTMU+PYP(I,10)
          ENDIF      
      ENDIF
      ENDDO
      
      DMLL(2)=PTE
      DMLL(3)=PTMU
      DMLL(4)=PTE+PTMU
      
      TOTLEP=EP+EN+MUP+MUN
      IARR(1)=EP
      IARR(2)=EN 
      IARR(4)=MUP
      IARR(5)=MUN      

      IF(TOTLEP .EQ. 2 ) THEN
        IF(EP .EQ. 1 .AND. EN .EQ. 1) THEN
	  DOTPR=ELEC(1,1)*ELEC(2,1)+ELEC(1,2)*ELEC(2,2)+ELEC(1,3)*ELEC(2,3)
          MLL=DSQRT(2.0D0*(ELEC(1,4)*ELEC(2,4)-DOTPR))
	  DMLL(1)=MLL
          IARR(3)=1 
	ENDIF

      	IF(MUP .EQ. 1 .AND. MUN .EQ. 1) THEN
	  DOTPR=MUON(1,1)*MUON(2,1)+MUON(1,2)*MUON(2,2)+MUON(1,3)*MUON(2,3)
          MLL=DSQRT(2.0D0*(MUON(1,4)*MUON(2,4)-DOTPR))
	  DMLL(1)=MLL
          IARR(3)=1          
	ENDIF
      ENDIF	
      END
C====================================================================




